// API Configuration for MFT GROUP
// This file provides a consistent way to construct API URLs regardless of deployment location

// Determine the base path for API calls
function getApiBaseUrl() {
    // Check if we're in a subdirectory like /MFT/
    const pathname = window.location.pathname;
    const origin = window.location.origin;
    const host = window.location.host;
    const protocol = window.location.protocol;
    console.log('=== API CONFIG DEBUG ===');
    console.log('Current pathname:', pathname);
    console.log('Current origin:', origin);
    console.log('Current host:', host);
    console.log('Current protocol:', protocol);
    
    // Handle file:// protocol (when opening HTML files directly)
    if (protocol === 'file:') {
        console.log('USING FILE PROTOCOL HANDLING');
        // When opening files directly, we need to use localhost
        // Check if we're in a typical MFT structure
        if (pathname.toLowerCase().includes('/mft/')) {
            console.log('RETURNING: http://localhost/MFT/api (file protocol with MFT)');
            return 'http://localhost/MFT/api';
        } else {
            console.log('RETURNING: http://localhost/api (file protocol without MFT)');
            return 'http://localhost/api';
        }
    }
    
    // Handle XAMPP localhost development (port 80/443)
    if (host.includes('localhost')) {
        console.log('USING XAMPP LOCALHOST HANDLING');
        // For XAMPP, the MFT directory is directly in htdocs
        // Use the actual case from the URL to avoid case sensitivity issues
        const pathname = window.location.pathname;
        let baseUrl;
        
        // Check if the URL contains /mft/ (case insensitive)
        if (pathname.toLowerCase().includes('/mft/')) {
            // Extract the actual case from the URL
            const pathParts = pathname.split('/');
            let mftPart = 'MFT'; // Default to uppercase
            
            for (let i = 0; i < pathParts.length; i++) {
                if (pathParts[i].toLowerCase() === 'mft') {
                    mftPart = pathParts[i]; // Use the actual case from the URL
                    break;
                }
            }
            
            // Use the same protocol as the frontend to avoid CORS issues
            baseUrl = protocol + '//' + host + '/' + mftPart + '/api';
        } else {
            // If no /mft/ in path, assume we're at the root
            // Use the same protocol as the frontend to avoid CORS issues
            baseUrl = protocol + '//' + host + '/api';
        }
        
        console.log('RETURNING:', baseUrl, '(XAMPP path with correct case and protocol)');
        return baseUrl;
    }
    
    // Handle localhost development server (port 8000)
    if (host.includes('localhost:8000')) {
        console.log('USING LOCALHOST:8000 HANDLING');
        return protocol + '://localhost:8000/api';
    }
    
    // For cPanel deployments, the path might be different
    // Check if we're in a typical cPanel structure
    if (pathname.includes('/public_html/')) {
        console.log('USING CPANEL HANDLING');
        // cPanel typically serves from public_html
        const baseUrl = protocol + '://' + host + '/api';
        console.log('RETURNING:', baseUrl, '(cPanel path)');
        return baseUrl;
    }
    
    // SPECIAL CASE FOR MFT DEPLOYMENT:
    // When the web server root is the MFT directory but accessed via /mft/ URL
    // We need to detect this scenario and return the correct API path
    if (pathname.toLowerCase().startsWith('/mft/')) {
        console.log('USING MFT DEPLOYMENT HANDLING');
        // Check if this is a direct MFT deployment where the web root is the MFT directory
        // In this case, the API is at the same level as the web root
        const baseUrl = protocol + '://' + host + '/api';
        console.log('RETURNING:', baseUrl, '(MFT deployment path)');
        return baseUrl;
    }
    
    // Check if we're in a development environment with MFT subdirectory
    if (pathname.toLowerCase().includes('/mft/')) {
        console.log('USING MFT SUBDIRECTORY HANDLING');
        // Determine the actual case of the MFT directory from the URL
        const pathParts = pathname.split('/');
        for (let i = 0; i < pathParts.length; i++) {
            if (pathParts[i].toLowerCase() === 'mft') {
                // SPECIAL FIX: If the web server root is the MFT directory, 
                // then we don't need to add another /mft/ to the path
                const baseUrl = protocol + '://' + host + '/api';
                console.log('RETURNING:', baseUrl, '(MFT subdirectory special case)');
                return baseUrl;
            }
        }
        // Fallback if we couldn't determine the case
        console.log('RETURNING: ' + protocol + '://localhost/api (fallback MFT path)');
        return protocol + '://localhost/api';
    }
    
    // Also check for other possible variations at the end of the path
    const pathVariations = ['/mft', '/Mft', '/MFT'];
    for (let variation of pathVariations) {
        if (pathname.toLowerCase().endsWith(variation.toLowerCase())) {
            const baseUrl = protocol + '://' + host + '/api';
            console.log('RETURNING:', baseUrl, '(path variation at end)');
            return baseUrl;
        }
    }
    
    // Check if MFT appears anywhere in the path
    const pathSegments = pathname.split('/');
    for (let i = 0; i < pathSegments.length; i++) {
        if (pathSegments[i].toLowerCase() === 'mft' && i < pathSegments.length - 1) {
            const baseUrl = protocol + '://' + host + '/api';
            console.log('RETURNING:', baseUrl, '(MFT in path segments)');
            return baseUrl;
        }
    }
    
    // Default to root path with api
    console.log('USING DEFAULT HANDLING');
    const baseUrl = protocol + '://' + host + '/api';
    console.log('RETURNING:', baseUrl, '(default path)');
    return baseUrl;
}

// Construct the full API URL for a given endpoint
function getApiUrl(endpoint) {
    // Remove leading slash from endpoint if present
    const cleanEndpoint = endpoint.startsWith('/') ? endpoint.substring(1) : endpoint;
    const baseUrl = getApiBaseUrl();
    console.log('Base URL:', baseUrl, 'Endpoint:', cleanEndpoint);
    
    // Ensure there's no double slash
    if (baseUrl.endsWith('/')) {
        return baseUrl + cleanEndpoint;
    } else {
        return baseUrl + '/' + cleanEndpoint;
    }
}

// Export functions for use in other modules
window.getApiBaseUrl = getApiBaseUrl;
window.getApiUrl = getApiUrl;

// Log the final result when the script loads
console.log('=== API CONFIG LOADED ===');
console.log('Final API Base URL:', getApiBaseUrl());