// MFT GROUP API Client
// This script provides functions to fetch data from the backend API

// Use the getApiBaseUrl function if available, otherwise default to '/api'
let API_BASE_URL;
if (typeof getApiBaseUrl === 'function') {
    API_BASE_URL = getApiBaseUrl();
} else {
    // Fallback to a more intelligent default based on the current URL
    const pathname = window.location.pathname;
    // For cPanel deployments
    if (pathname.includes('/public_html/')) {
        API_BASE_URL = window.location.origin + '/api';
    } else if (pathname.toLowerCase().includes('/mft/')) {
        // Key fix: Properly detect MFT subdirectory
        API_BASE_URL = window.location.origin + '/MFT/api';
    } else {
        API_BASE_URL = window.location.origin + '/api';
    }
}
console.log('API Base URL set to:', API_BASE_URL);

// Function to get member data
async function getMemberData(memberId) {
    try {
        const url = `${API_BASE_URL}/members.php?member_id=${encodeURIComponent(memberId)}`;
        console.log('Fetching member data from:', url);
        
        // Add timeout to prevent hanging requests
        const controller = new AbortController();
        const timeoutId = setTimeout(() => controller.abort(), 10000); // 10 second timeout
        
        const response = await fetch(url, {
            signal: controller.signal
        });
        
        clearTimeout(timeoutId);
        
        if (!response.ok) {
            const errorText = await response.text();
            console.error('Member API error response:', errorText);
            throw new Error(`HTTP error! status: ${response.status}, message: ${errorText}`);
        }
        const contentType = response.headers.get('content-type');
        if (!contentType || !contentType.includes('application/json')) {
            const errorText = await response.text();
            console.error('Non-JSON response from member API:', errorText);
            throw new Error('Response is not JSON: ' + errorText);
        }
        const data = await response.json();
        console.log('Member API response:', data);
        return data;
    } catch (error) {
        console.error('Error fetching member data:', error);
        return { error: error.message };
    }
}

// Function to get member contributions
async function getMemberContributions(memberId) {
    try {
        // Log the memberId being used
        console.log('getMemberContributions called with memberId:', memberId);
        
        // Ensure memberId is provided
        if (!memberId) {
            throw new Error('Member ID is required');
        }
        
        const url = `${API_BASE_URL}/contributions.php?member_id=${encodeURIComponent(memberId)}`;
        console.log('Fetching contributions data from:', url);
        
        // Add timeout to prevent hanging requests
        const controller = new AbortController();
        const timeoutId = setTimeout(() => controller.abort(), 10000); // 10 second timeout
        
        const response = await fetch(url, {
            signal: controller.signal
        });
        
        clearTimeout(timeoutId);
        
        console.log('Contributions API response status:', response.status);
        
        if (!response.ok) {
            const errorText = await response.text();
            console.error('API Error response:', errorText);
            throw new Error(`HTTP error! status: ${response.status}, message: ${errorText}`);
        }
        
        const contentType = response.headers.get('content-type');
        if (!contentType || !contentType.includes('application/json')) {
            const errorText = await response.text();
            console.error('Non-JSON response:', errorText);
            throw new Error('Response is not JSON: ' + errorText);
        }
        
        const data = await response.json();
        console.log('Contributions API response:', data);
        return data;
    } catch (error) {
        console.error('Error fetching contributions data:', error);
        return { error: error.message };
    }
}

// Function to get member loans
async function getMemberLoans(memberId) {
    try {
        const url = `${API_BASE_URL}/loans.php?member_id=${encodeURIComponent(memberId)}`;
        console.log('Fetching loans data from:', url);
        
        // Add timeout to prevent hanging requests
        const controller = new AbortController();
        const timeoutId = setTimeout(() => controller.abort(), 10000); // 10 second timeout
        
        const response = await fetch(url, {
            signal: controller.signal
        });
        
        clearTimeout(timeoutId);
        
        if (!response.ok) {
            const errorText = await response.text();
            console.error('Loans API error response:', errorText);
            throw new Error(`HTTP error! status: ${response.status}, message: ${errorText}`);
        }
        const contentType = response.headers.get('content-type');
        if (!contentType || !contentType.includes('application/json')) {
            const errorText = await response.text();
            console.error('Non-JSON response from loans API:', errorText);
            throw new Error('Response is not JSON: ' + errorText);
        }
        const data = await response.json();
        console.log('Loans API response:', data);
        return data;
    } catch (error) {
        console.error('Error fetching loans data:', error);
        return { error: error.message };
    }
}

// Function to get member fines
async function getMemberFines(memberId) {
    try {
        const url = `${API_BASE_URL}/fines.php?member_id=${encodeURIComponent(memberId)}`;
        console.log('Fetching fines data from:', url);
        
        // Add timeout to prevent hanging requests
        const controller = new AbortController();
        const timeoutId = setTimeout(() => controller.abort(), 10000); // 10 second timeout
        
        const response = await fetch(url, {
            signal: controller.signal
        });
        
        clearTimeout(timeoutId);
        
        if (!response.ok) {
            const errorText = await response.text();
            console.error('Fines API error response:', errorText);
            throw new Error(`HTTP error! status: ${response.status}, message: ${errorText}`);
        }
        const contentType = response.headers.get('content-type');
        if (!contentType || !contentType.includes('application/json')) {
            const errorText = await response.text();
            console.error('Non-JSON response from fines API:', errorText);
            throw new Error('Response is not JSON: ' + errorText);
        }
        const data = await response.json();
        console.log('Fines API response:', data);
        return data;
    } catch (error) {
        console.error('Error fetching fines data:', error);
        return { error: error.message };
    }
}

// Function to get member projects
async function getMemberProjects(memberId) {
    try {
        const url = `${API_BASE_URL}/projects.php?member_id=${encodeURIComponent(memberId)}`;
        console.log('Fetching projects data from:', url);
        
        // Add timeout to prevent hanging requests
        const controller = new AbortController();
        const timeoutId = setTimeout(() => controller.abort(), 10000); // 10 second timeout
        
        const response = await fetch(url, {
            signal: controller.signal
        });
        
        clearTimeout(timeoutId);
        
        // Log response status for debugging
        console.log('Projects API response status:', response.status);
        
        if (!response.ok) {
            const errorText = await response.text();
            console.error('API Error response:', errorText);
            throw new Error(`HTTP error! status: ${response.status}, message: ${errorText}`);
        }
        
        const contentType = response.headers.get('content-type');
        if (!contentType || !contentType.includes('application/json')) {
            const errorText = await response.text();
            console.error('Non-JSON response:', errorText);
            throw new Error('Response is not JSON: ' + errorText);
        }
        
        const data = await response.json();
        console.log('Projects API response:', data);
        return data;
    } catch (error) {
        console.error('Error fetching projects data:', error);
        return { error: error.message };
    }
}

// Function to get member dashboard data
async function getMemberDashboardData(memberId) {
    try {
        const url = `${API_BASE_URL}/dashboard.php?member_id=${encodeURIComponent(memberId)}&type=member`;
        console.log('Fetching dashboard data from:', url);
        
        // Add timeout to prevent hanging requests
        const controller = new AbortController();
        const timeoutId = setTimeout(() => controller.abort(), 15000); // 15 second timeout
        
        const response = await fetch(url, {
            signal: controller.signal
        });
        
        clearTimeout(timeoutId);
        
        if (!response.ok) {
            const errorText = await response.text();
            console.error('Dashboard API error response:', errorText);
            throw new Error(`HTTP error! status: ${response.status}, message: ${errorText}`);
        }
        const contentType = response.headers.get('content-type');
        if (!contentType || !contentType.includes('application/json')) {
            const errorText = await response.text();
            console.error('Non-JSON response from dashboard API:', errorText);
            throw new Error('Response is not JSON: ' + errorText);
        }
        const data = await response.json();
        console.log('Dashboard API response:', data);
        return data;
    } catch (error) {
        console.error('Error fetching dashboard data:', error);
        return { error: error.message };
    }
}

// Function to get all contributions with pagination
async function getAllContributions(page = 1, limit = 10) {
    try {
        const url = `${API_BASE_URL}/contributions.php?limit=${limit}&offset=${(page - 1) * limit}`;
        console.log('Fetching all contributions data from:', url);
        const response = await fetch(url);
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        const contentType = response.headers.get('content-type');
        if (!contentType || !contentType.includes('application/json')) {
            throw new Error('Response is not JSON');
        }
        return await response.json();
    } catch (error) {
        console.error('Error fetching contributions data:', error);
        return null;
    }
}

// Export functions
window.mftApi = {
    getMemberData,
    getMemberContributions,
    getMemberLoans,
    getMemberFines,
    getMemberProjects,
    getMemberDashboardData,
    getAllContributions
};