// MFT GROUP Admin Dashboard Real-Time Updates
// This script handles real-time updates specifically for admin dashboard stats

console.log('=== ADMIN DASHBOARD REAL-TIME UPDATES LOADING ===');

// Function to update admin dashboard with real-time data
function updateAdminDashboardRealTime(data) {
    console.log('Updating admin dashboard with real-time data:', data);
    
    // Handle different types of notifications
    if (data.type === 'member_update') {
        handleMemberUpdate(data.data);
    } else if (data.type === 'contribution_update') {
        handleContributionUpdate(data.data);
    } else if (data.type === 'loan_update') {
        handleLoanUpdate(data.data);
    } else if (data.type === 'fine_update') {
        handleFineUpdate(data.data);
    } else if (data.type === 'settings_update') {
        handleSettingsUpdate(data.data);
    } else if (data.type === 'project_update') {
        handleProjectUpdate(data.data);
    } else if (data.type === 'dashboard_update') {
        handleGeneralDashboardUpdate(data.data);
    } else {
        // Handle general dashboard updates
        handleGeneralDashboardUpdate(data);
    }
    
    // Show notification that data was updated
    showNotification('Dashboard updated with latest information', 'success');
}

// Handle member updates
function handleMemberUpdate(data) {
    console.log('Processing member update:', data);
    
    // Update members card
    if (data.action) {
        updateMembersCard(data.action);
    }
    
    // Show notification
    showNotification(`Member ${data.action} successfully`, 'success');
}

// Handle contribution updates
function handleContributionUpdate(data) {
    console.log('Processing contribution update:', data);
    
    // Update contributions card
    if (data.amount !== undefined) {
        updateContributionsCard(data.amount);
    }
    
    // Show notification
    showNotification(`New contribution recorded: KSH ${Number(data.amount).toLocaleString()}`, 'success');
}

// Handle loan updates
function handleLoanUpdate(data) {
    console.log('Processing loan update:', data);
    
    // Update loans card
    if (data.amount !== undefined) {
        updateLoansCard(data.amount);
    }
    
    // Show notification
    if (data.action === 'new_loan') {
        showNotification(`New loan applied for member: KSH ${Number(data.amount).toLocaleString()}`, 'success');
    } else {
        showNotification(`Loan ${data.status}: KSH ${Number(data.amount).toLocaleString()}`, 'info');
    }
    
    // Refresh loans section if on loans page
    if (typeof refreshLoansSection === 'function') {
        refreshLoansSection();
    }
}

// Handle fine updates
function handleFineUpdate(data) {
    console.log('Processing fine update:', data);
    
    // Update fines card
    if (data.amount !== undefined) {
        updateFinesCard(data.amount);
    }
    
    // Show notification
    showNotification(`Fine ${data.status}: KSH ${Number(data.amount).toLocaleString()}`, 'info');
}

// Handle settings updates
function handleSettingsUpdate(data) {
    console.log('Processing settings update:', data);
    
    // In a real implementation, you would update UI elements based on changed settings
    showNotification('System settings have been updated', 'info');
}

// Handle project updates
function handleProjectUpdate(data) {
    console.log('Processing project update:', data);
    
    // Update projects card
    if (data.action) {
        updateProjectsCard(data.action);
    }
    
    // Show notification
    showNotification(`Project ${data.action} successfully`, 'info');
}

// Handle general dashboard updates
function handleGeneralDashboardUpdate(data) {
    console.log('Processing general dashboard update:', data);
    
    // Update admin info in header
    if (data.admin_info) {
        updateAdminHeader(data.admin_info);
    }
    
    // Update contributions card
    if (data.contributions && data.contributions.total_balance !== undefined) {
        updateContributionsCard(data.contributions.total_balance);
    }
    
    // Update loans card
    if (data.loans && data.loans.outstanding_balance !== undefined) {
        updateLoansCard(data.loans.outstanding_balance);
    }
    
    // Update fines card
    if (data.fines && data.fines.total_fines !== undefined) {
        updateFinesCard(data.fines.total_fines);
    }
    
    // Update members card
    if (data.members) {
        updateMembersCard(data.members);
    }
    
    // Update projects card
    if (data.projects) {
        updateProjectsCard(data.projects);
    }
    
    // Update bank balance card
    if (data.bank_balance !== undefined) {
        updateBankBalanceCard(data.bank_balance);
    }
    
    // Update expenses card
    if (data.expenses !== undefined) {
        updateExpensesCard(data.expenses);
    }
    
    // Update net balance card
    if (data.net_balance !== undefined) {
        updateNetBalanceCard(data.net_balance);
    }
    
    // Update fines paid card
    if (data.paid_fines !== undefined) {
        updateFinesPaidCard(data.paid_fines);
    }
    
    // Update financial overview if available
    if (data.financial_summary) {
        updateFinancialOverview(data.financial_summary);
    }
}

// Function to update admin header information
function updateAdminHeader(adminInfo) {
    // Update greeting
    const greetingElement = document.getElementById('greeting');
    if (greetingElement) {
        // Get appropriate greeting based on time of day
        const greeting = getGreetingBasedOnTime();
        greetingElement.textContent = `${greeting}, ${adminInfo.name || 'Admin'}`;
    }
    
    // Update admin name
    const adminNameElement = document.getElementById('adminName');
    if (adminNameElement) {
        adminNameElement.textContent = adminInfo.name || 'Admin User';
    }
}

// Function to get appropriate greeting based on time of day
function getGreetingBasedOnTime() {
    const hours = new Date().getHours();
    
    if (hours >= 5 && hours < 12) {
        return 'Good morning';
    } else if (hours >= 12 && hours < 18) {
        return 'Good afternoon';
    } else {
        return 'Good evening';
    }
}

// Function to update contributions card
function updateContributionsCard(balance) {
    if (balance !== undefined) {
        // Find all contributions cards
        const contributionCards = document.querySelectorAll('.stat-card.contributions');
        contributionCards.forEach(card => {
            const valueElement = card.querySelector('.stat-value');
            if (valueElement) {
                valueElement.textContent = `KSH ${Number(balance).toLocaleString()}`;
            }
        });
    }
}

// Function to update loans card
function updateLoansCard(balance) {
    if (balance !== undefined) {
        // Find all loans cards
        const loanCards = document.querySelectorAll('.stat-card.loans');
        loanCards.forEach(card => {
            const valueElement = card.querySelector('.stat-value');
            if (valueElement) {
                valueElement.textContent = `KSH ${Number(balance).toLocaleString()}`;
            }
        });
    }
}

// Function to update fines card
function updateFinesCard(balance) {
    if (balance !== undefined) {
        // Find all fines cards
        const finesCards = document.querySelectorAll('.stat-card.fines');
        finesCards.forEach(card => {
            const valueElement = card.querySelector('.stat-value');
            if (valueElement) {
                valueElement.textContent = `KSH ${Number(balance).toLocaleString()}`;
            }
        });
    }
}

// Function to update members card
function updateMembersCard(membersData) {
    // Update the members card with real-time data
    const membersCards = document.querySelectorAll('.stat-card.members');
    membersCards.forEach(card => {
        const valueElement = card.querySelector('.stat-value');
        if (valueElement) {
            // If membersData is a number, use it directly
            // If it's an object with count, use that
            const count = typeof membersData === 'number' ? membersData : 
                         membersData.count !== undefined ? membersData.count : 
                         membersData.total !== undefined ? membersData.total : 
                         null;
            
            if (count !== null) {
                valueElement.textContent = count;
            }
        }
        
        // Update detail elements if membersData is an object
        if (typeof membersData === 'object' && membersData !== null) {
            const detailElements = card.querySelectorAll('.stat-detail');
            if (detailElements.length >= 2) {
                detailElements[0].textContent = `${membersData.active || 0} Active`;
                detailElements[1].textContent = `${membersData.pending || 0} Pending`;
            }
        }
    });
}

// Function to update projects card
function updateProjectsCard(projectsData) {
    // Update the projects card with real-time data
    const projectsCards = document.querySelectorAll('.stat-card.projects');
    projectsCards.forEach(card => {
        const valueElement = card.querySelector('.stat-value');
        if (valueElement) {
            // If projectsData is a number, use it directly
            // If it's an object with count, use that
            const count = typeof projectsData === 'number' ? projectsData : 
                         projectsData.count !== undefined ? projectsData.count : 
                         projectsData.active !== undefined ? projectsData.active : 
                         null;
            
            if (count !== null) {
                valueElement.textContent = count;
            }
        }
        
        // Update detail elements if projectsData is an object
        if (typeof projectsData === 'object' && projectsData !== null && projectsData.completed !== undefined) {
            const detailElement = card.querySelector('.stat-detail:last-child');
            if (detailElement) {
                detailElement.textContent = `${projectsData.completed} Completed`;
            }
        }
    });
}

// Function to update bank balance card
function updateBankBalanceCard(balance) {
    if (balance !== undefined) {
        // Find all bank balance cards
        const bankCards = document.querySelectorAll('.stat-card.bank');
        bankCards.forEach(card => {
            const valueElement = card.querySelector('.stat-value');
            if (valueElement) {
                valueElement.textContent = `KSH ${Number(balance).toLocaleString()}`;
            }
        });
    }
}

// Function to update expenses card
function updateExpensesCard(expenses) {
    if (expenses !== undefined) {
        // Find all expenses cards
        const expenseCards = document.querySelectorAll('.stat-card.expenses');
        expenseCards.forEach(card => {
            const valueElement = card.querySelector('.stat-value');
            if (valueElement) {
                valueElement.textContent = `KSH ${Number(expenses).toLocaleString()}`;
            }
        });
    }
}

// Function to update net balance card
function updateNetBalanceCard(netBalance) {
    if (netBalance !== undefined) {
        // Find all net balance cards
        const netBalanceCards = document.querySelectorAll('.stat-card.net-balance');
        netBalanceCards.forEach(card => {
            const valueElement = card.querySelector('.stat-value');
            if (valueElement) {
                valueElement.textContent = `KSH ${Number(netBalance).toLocaleString()}`;
            }
        });
    }
}

// Function to update fines paid card
function updateFinesPaidCard(paidFines) {
    if (paidFines !== undefined) {
        // Find all fines paid cards
        const finesPaidCards = document.querySelectorAll('.stat-card.fines-paid');
        finesPaidCards.forEach(card => {
            const valueElement = card.querySelector('.stat-value');
            if (valueElement) {
                valueElement.textContent = `KSH ${Number(paidFines).toLocaleString()}`;
            }
        });
    }
}

// Function to update financial overview
function updateFinancialOverview(financialData) {
    console.log('Updating financial overview with data:', financialData);
    const financialContainer = document.getElementById('financialOverviewContainer');
    if (financialContainer && financialData && financialData.accounts) {
        financialContainer.innerHTML = '';
        
        if (financialData.accounts.length === 0) {
            financialContainer.innerHTML = '<div class="account-item"><div class="account-name">No accounts found</div></div>';
            return;
        }
        
        financialData.accounts.forEach(account => {
            const accountElement = document.createElement('div');
            accountElement.className = 'account-item';
            const balance = parseFloat(account.balance) || 0;
            accountElement.innerHTML = `
                <div class="account-name">${account.name || 'Unnamed Account'}</div>
                <div class="account-balance">KSH ${balance.toLocaleString()}</div>
            `;
            financialContainer.appendChild(accountElement);
        });
    }
}

// Function to show a notification to the user
function showNotification(message, type = 'info') {
    // Check if notification already exists
    const existingNotification = document.querySelector('.notification');
    if (existingNotification) {
        existingNotification.remove();
    }
    
    // Create a notification element
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.textContent = message;
    notification.style.cssText = `
        position: fixed;
        top: 80px;
        right: 20px;
        padding: 15px 20px;
        border-radius: 5px;
        color: white;
        background-color: ${type === 'success' ? '#4CAF50' : type === 'error' ? '#F44336' : '#2196F3'};
        box-shadow: 0 4px 8px rgba(0,0,0,0.2);
        z-index: 10000;
        font-family: 'Inter', sans-serif;
        transform: translateX(120%);
        transition: transform 0.3s ease;
    `;
    
    document.body.appendChild(notification);
    
    // Trigger reflow to enable transition
    notification.offsetHeight;
    notification.style.transform = 'translateX(0)';
    
    // Remove the notification after 3 seconds
    setTimeout(() => {
        notification.style.transform = 'translateX(120%)';
        setTimeout(() => {
            if (notification.parentNode) {
                notification.parentNode.removeChild(notification);
            }
        }, 300);
    }, 3000);
}

// Function to manually refresh the dashboard
function refreshAdminDashboard() {
    // Trigger an immediate update
    if (typeof mftRealTime !== 'undefined' && typeof mftRealTime.pollForUpdates === 'function') {
        mftRealTime.pollForUpdates();
    } else if (typeof loadAdminDashboardData === 'function') {
        // Fallback to loading data directly
        loadAdminDashboardData();
    }
}

// Export functions for use in other modules
window.updateAdminDashboardRealTime = updateAdminDashboardRealTime;
window.refreshAdminDashboard = refreshAdminDashboard;