// MFT GROUP Admin Dashboard Data Loader - FIXED VERSION
// This script fetches and displays admin data on the dashboard

console.log('=== ADMIN DASHBOARD DATA LOADER LOADING ===');

// Ensure we have the API functions available
if (typeof getApiUrl !== 'function') {
    console.log('Using fallback implementation for getApiUrl');
    // Fallback implementation if getApiUrl is not available
    function getApiUrl(endpoint) {
        console.log('Fallback getApiUrl called with endpoint:', endpoint);
        // Use the same logic as in api-config.js
        const pathname = window.location.pathname;
        const origin = window.location.origin;
        let baseUrl;
        
        console.log('Fallback URL construction - Pathname:', pathname);
        console.log('Fallback URL construction - Origin:', origin);
        
        // SPECIAL CASE FOR MFT DEPLOYMENT:
        // When the web server root is the MFT directory but accessed via /mft/ URL
        if (pathname.toLowerCase().startsWith('/mft/')) {
            baseUrl = origin + '/api';
        } else if (pathname.toLowerCase().includes('/mft/')) {
            // Determine the actual case of the MFT directory from the URL
            const pathParts = pathname.split('/');
            for (let i = 0; i < pathParts.length; i++) {
                if (pathParts[i].toLowerCase() === 'mft') {
                    baseUrl = origin + '/api';
                    break;
                }
            }
            // Fallback if we couldn't determine the case
            if (!baseUrl) {
                baseUrl = origin + '/api';
            }
        } else {
            baseUrl = origin + '/api';
        }
        
        console.log('Fallback URL construction - Base URL:', baseUrl);
        
        // Remove leading slash from endpoint if present
        const cleanEndpoint = endpoint.startsWith('/') ? endpoint.substring(1) : endpoint;
        
        // Ensure there's no double slash
        if (baseUrl.endsWith('/')) {
            return baseUrl + cleanEndpoint;
        } else {
            return baseUrl + '/' + cleanEndpoint;
        }
    }
} else {
    console.log('Using imported getApiUrl function');
}

// Function to get user role from session storage
function getUserRole() {
    try {
        const adminData = sessionStorage.getItem('adminData');
        if (adminData) {
            const admin = JSON.parse(adminData);
            return admin.role || 'Member';
        }
    } catch (e) {
        console.error('Error getting user role:', e);
    }
    return 'Member';
}

// Function to check if user has permission for a specific feature
function hasPermission(feature) {
    // DISABLED: All users now have full access to all features
    return true;
    
    /*
    const role = getUserRole();
    
    // Define role permissions
    const rolePermissions = {
        'Admin': ['all'],
        'Treasurer': ['finance', 'reports', 'contributions', 'loans'],
        'Secretary': ['members', 'documents', 'meetings', 'finance'],
        'Chairman': ['members', 'meetings', 'reports', 'finance', 'contributions', 'loans'],
        'Assistant Chairman': ['members', 'meetings', 'reports', 'finance', 'contributions', 'loans'],
        'Assistant Secretary': ['members', 'documents', 'meetings', 'finance'],
        'Loans Officer': ['loans', 'reports', 'finance'],
        'Member': ['profile', 'contributions', 'loans', 'documents', 'finance']
    };
    
    const permissions = rolePermissions[role] || [];
    
    // Admin has access to everything
    if (permissions.includes('all')) {
        return true;
    }
    
    return permissions.includes(feature);
    */
}

// Function to load admin dashboard data
async function loadAdminDashboardData() {
    console.log('Loading admin dashboard data');
    try {
        // Load dashboard statistics
        const dashboardData = await getAdminDashboardStats();
        if (dashboardData && !dashboardData.error) {
            updateAdminDashboardCards(dashboardData);
        } else if (dashboardData && dashboardData.error) {
            console.error('API returned error:', dashboardData.error);
        }
        
        // Load recent activities
        const activitiesData = await getRecentActivities();
        if (activitiesData && !activitiesData.error) {
            updateRecentActivities(activitiesData);
        }
        
        // Load financial overview
        const financialData = await getFinancialOverview();
        if (financialData && !financialData.error) {
            updateFinancialOverview(financialData);
        }
    } catch (error) {
        console.error('Error loading admin dashboard data:', error);
        // Show error to user
        showErrorMessage('Failed to load dashboard data. Please try again.');
    }
}

// Cache for dashboard data
const dashboardDataCache = {
    data: null,
    timestamp: 0,
    expirationTime: 30 * 60 * 1000 // Extend cache expiration to 30 minutes
};

// Optimized version that loads all data in a single API call with caching
async function loadAdminDashboardDataOptimized() {
    console.log('Loading optimized admin dashboard data');
    
    // Check if we have valid cached data that's not expired
    const now = Date.now();
    if (dashboardDataCache.data && (now - dashboardDataCache.timestamp) < dashboardDataCache.expirationTime) {
        console.log('Using cached dashboard data');
        const cachedData = dashboardDataCache.data;
        
        // Update all dashboard components with the cached data
        if (cachedData.stats) {
            updateAdminDashboardCards(cachedData.stats);
        }
        
        if (cachedData.activities) {
            updateRecentActivities(cachedData.activities);
        }
        
        if (cachedData.financial) {
            updateFinancialOverview(cachedData.financial);
        }
        
        return;
    }
    
    try {
        // Single API call to get all dashboard data
        const endpoint = 'dashboard.php?action=getAllAdminData';
        const url = getApiUrl(endpoint);
        console.log('Fetching all admin data from:', url);
        
        const response = await fetch(url);
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        
        const contentType = response.headers.get('content-type');
        if (!contentType || !contentType.includes('application/json')) {
            throw new Error('Response is not JSON');
        }
        
        const data = await response.json();
        console.log('All admin data received:', data);
        
        // Cache the data
        dashboardDataCache.data = data;
        dashboardDataCache.timestamp = now;
        
        // Update all dashboard components with the received data
        if (data.stats) {
            updateAdminDashboardCards(data.stats);
        }
        
        if (data.activities) {
            updateRecentActivities(data.activities);
        }
        
        if (data.financial) {
            updateFinancialOverview(data.financial);
        }
        
    } catch (error) {
        console.error('Error loading optimized admin dashboard data:', error);
        // Fallback to original method
        console.log('Falling back to original data loading method');
        await loadAdminDashboardData();
    }
}

// Function to get admin dashboard statistics
async function getAdminDashboardStats() {
    try {
        const endpoint = 'dashboard.php?action=getAdminStats';
        const url = getApiUrl(endpoint);
        console.log('Fetching admin stats from:', url);
        
        // Fetch real data from the API using the correct URL function
        const response = await fetch(url);
        console.log('Admin stats response status:', response.status);
        
        if (!response.ok) {
            const text = await response.text();
            console.error('Admin stats error response:', text.substring(0, 200));
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        
        const contentType = response.headers.get('content-type');
        console.log('Admin stats content type:', contentType);
        
        if (!contentType || !contentType.includes('application/json')) {
            const text = await response.text();
            console.error('Admin stats response is not JSON:', text.substring(0, 200));
            throw new Error('Response is not JSON');
        }
        
        const data = await response.json();
        console.log('Admin stats data received:', data);
        return data;
    } catch (error) {
        console.error('Error fetching admin dashboard stats:', error);
        return {error: error.message};
    }
}

// Function to get recent activities
async function getRecentActivities() {
    try {
        const endpoint = 'dashboard.php?action=getRecentActivities';
        const url = getApiUrl(endpoint);
        console.log('Fetching recent activities from:', url);
        
        // Fetch real data from the API using the correct URL function
        const response = await fetch(url);
        console.log('Recent activities response status:', response.status);
        
        if (!response.ok) {
            const text = await response.text();
            console.error('Recent activities error response:', text.substring(0, 200));
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        
        const contentType = response.headers.get('content-type');
        console.log('Recent activities content type:', contentType);
        
        if (!contentType || !contentType.includes('application/json')) {
            const text = await response.text();
            console.error('Recent activities response is not JSON:', text.substring(0, 200));
            throw new Error('Response is not JSON');
        }
        
        return await response.json();
    } catch (error) {
        console.error('Error fetching recent activities:', error);
        return null;
    }
}

// Function to get financial overview
async function getFinancialOverview() {
    try {
        const endpoint = 'dashboard.php?action=getFinancialOverview';
        const url = getApiUrl(endpoint);
        console.log('Fetching financial overview from:', url);
        
        // Fetch real data from the API using the correct URL function
        const response = await fetch(url);
        console.log('Financial overview response status:', response.status);
        
        if (!response.ok) {
            const text = await response.text();
            console.error('Financial overview error response:', text.substring(0, 200));
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        
        const contentType = response.headers.get('content-type');
        console.log('Financial overview content type:', contentType);
        
        if (!contentType || !contentType.includes('application/json')) {
            const text = await response.text();
            console.error('Financial overview response is not JSON:', text.substring(0, 200));
            throw new Error('Response is not JSON');
        }
        
        return await response.json();
    } catch (error) {
        console.error('Error fetching financial overview:', error);
        return null;
    }
}

// Function to update admin dashboard cards with improved error handling and formatting
function updateAdminDashboardCards(data) {
    console.log('Updating admin dashboard cards with data:', data);
    
    // Check if data is valid
    if (!data) {
        console.error('No data provided to update dashboard cards');
        return;
    }
    
    // Get user role
    const userRole = getUserRole();
    console.log('User role:', userRole);
    
    // NEW: All admins should see dashboard overview (view-only)
    // The dashboard cards will always be visible, but data will be restricted based on permissions
    
    // Update members card (always visible for all admins)
    const membersCard = document.getElementById('total-members');
    if (membersCard && data.members) {
        if (hasPermission('members')) {
            const totalMembers = data.members.total || 0;
            membersCard.textContent = totalMembers;
            console.log('Updated members card value to:', totalMembers);
        } else {
            // Show restricted access message with tooltip
            membersCard.innerHTML = '<span class="restricted-tooltip">Restricted<span class="tooltip-text">Contact Treasurer for access</span></span>';
            console.log('Restricted access to members data for role:', userRole);
        }
    }
    
    // Update contributions card (always visible but data restricted)
    const contributionsCard = document.getElementById('total-contributions');
    if (contributionsCard && data.contributions) {
        if (hasPermission('contributions')) {
            const balance = parseFloat(data.contributions.total_balance) || 0;
            contributionsCard.textContent = `KSH ${balance.toLocaleString()}`;
            console.log('Updated contributions card value to:', `KSH ${balance.toLocaleString()}`);
        } else {
            // Show restricted access message with tooltip
            contributionsCard.innerHTML = '<span class="restricted-tooltip">Restricted<span class="tooltip-text">Contact Treasurer for access</span></span>';
            console.log('Restricted access to contributions data for role:', userRole);
        }
    }
    
    // Update loans card (always visible but data restricted)
    const loansCard = document.getElementById('active-loans');
    if (loansCard && data.loans) {
        if (hasPermission('loans')) {
            const balance = parseFloat(data.loans.outstanding_balance) || 0;
            loansCard.textContent = `KSH ${balance.toLocaleString()}`;
            console.log('Updated loans card value to:', `KSH ${balance.toLocaleString()}`);
        } else {
            // Show restricted access message with tooltip
            loansCard.innerHTML = '<span class="restricted-tooltip">Restricted<span class="tooltip-text">Contact Loans Officer for access</span></span>';
            console.log('Restricted access to loans data for role:', userRole);
        }
    }
    
    // Update projects card (always visible for all admins)
    const projectsCard = document.getElementById('active-projects');
    if (projectsCard && data.projects) {
        const active = data.projects.active || 0;
        projectsCard.textContent = active;
        console.log('Updated projects card value to:', active);
    }
    
    // Update bank balance card (always visible but data restricted)
    const bankCard = document.getElementById('bank-balance');
    if (bankCard && data.financial_summary) {
        if (hasPermission('finance')) {
            // Bank Balance = Total Contributions + Fines Paid
            const balance = parseFloat(data.financial_summary.bank_balance) || 0;
            bankCard.textContent = `KSH ${balance.toLocaleString()}`;
            console.log('Updated bank card value to:', `KSH ${balance.toLocaleString()}`);
        } else {
            // Show restricted access message with tooltip
            bankCard.innerHTML = '<span class="restricted-tooltip">Restricted<span class="tooltip-text">Contact Treasurer for access</span></span>';
            console.log('Restricted access to bank balance data for role:', userRole);
        }
    }
    
    // Update pending fines card (always visible but data restricted)
    const finesCard = document.getElementById('pending-fines');
    if (finesCard && data.fines) {
        if (hasPermission('finance')) {
            const balance = parseFloat(data.fines.total_pending_fines) || 0;
            finesCard.textContent = `KSH ${balance.toLocaleString()}`;
            console.log('Updated fines card value to:', `KSH ${balance.toLocaleString()}`);
        } else {
            // Show restricted access message with tooltip
            finesCard.innerHTML = '<span class="restricted-tooltip">Restricted<span class="tooltip-text">Contact Treasurer for access</span></span>';
            console.log('Restricted access to pending fines data for role:', userRole);
        }
    }
    
    // Update fines paid card (always visible but data restricted)
    const finesPaidCard = document.getElementById('fines-paid');
    if (finesPaidCard && data.paid_fines) {
        if (hasPermission('finance')) {
            const balance = parseFloat(data.paid_fines.total_paid_fines) || 0;
            finesPaidCard.textContent = `KSH ${balance.toLocaleString()}`;
            console.log('Updated fines paid card value to:', `KSH ${balance.toLocaleString()}`);
        } else {
            // Show restricted access message with tooltip
            finesPaidCard.innerHTML = '<span class="restricted-tooltip">Restricted<span class="tooltip-text">Contact Treasurer for access</span></span>';
            console.log('Restricted access to fines paid data for role:', userRole);
        }
    }
    
    // Update expenses card (always visible but data restricted)
    const expensesCard = document.getElementById('total-expenses');
    if (expensesCard && data.financial_summary) {
        if (hasPermission('finance')) {
            // Total Expenses from Expense accounts
            const expenses = parseFloat(data.financial_summary.total_expenses) || 0;
            expensesCard.textContent = `KSH ${expenses.toLocaleString()}`;
            console.log('Updated expenses card value to:', `KSH ${expenses.toLocaleString()}`);
        } else {
            // Show restricted access message with tooltip
            expensesCard.innerHTML = '<span class="restricted-tooltip">Restricted<span class="tooltip-text">Contact Treasurer for access</span></span>';
            console.log('Restricted access to expenses data for role:', userRole);
        }
    }
    
    // Update net balance card (always visible but data restricted)
    const netBalanceCard = document.getElementById('net-balance');
    if (netBalanceCard && data.financial_summary) {
        if (hasPermission('finance')) {
            // Net Balance = (Total Contributions + Fines Paid) - Expenses
            const netBalance = parseFloat(data.financial_summary.net_balance) || 0;
            netBalanceCard.textContent = `KSH ${netBalance.toLocaleString()}`;
            console.log('Updated net balance card value to:', `KSH ${netBalance.toLocaleString()}`);
        } else {
            // Show restricted access message with tooltip
            netBalanceCard.innerHTML = '<span class="restricted-tooltip">Restricted<span class="tooltip-text">Contact Treasurer for access</span></span>';
            console.log('Restricted access to net balance data for role:', userRole);
        }
    }
}

// Function to update recent activities with optimized DOM manipulation
function updateRecentActivities(activities) {
    console.log('Updating recent activities with data:', activities);
    const activitiesContainer = document.getElementById('recentActivitiesContainer');
    if (activitiesContainer && activities && Array.isArray(activities)) {
        // Use innerHTML for better performance instead of creating elements one by one
        let html = '';
        
        if (activities.length === 0) {
            html = '<div class="activity-item"><div class="activity-content"><div class="activity-description">No recent activities</div></div></div>';
        } else {
            // Limit to 30 activities to prevent performance issues
            const limitedActivities = activities.slice(0, 30);
            limitedActivities.forEach(activity => {
                html += `
                    <div class="activity-item">
                        <div class="activity-icon">
                            <i class="fas fa-${getActivityIcon(activity.type)}"></i>
                        </div>
                        <div class="activity-content">
                            <div class="activity-description">${activity.description || 'No description'}</div>
                            <div class="activity-time">${formatActivityTime(activity.timestamp)}</div>
                        </div>
                    </div>
                `;
            });
        }
        
        // Replace content in a single DOM operation
        activitiesContainer.innerHTML = html;
    }
}

// Function to update financial overview
function updateFinancialOverview(financialData) {
    console.log('Updating financial overview with data:', financialData);
    // Note: There's no financialOverviewContainer in the HTML, so we skip this for now
    // This function is kept for compatibility but doesn't do anything
}

// Helper function to get activity icon based on type
function getActivityIcon(type) {
    switch (type) {
        case 'contribution':
            return 'wallet';
        case 'loan':
            return 'credit-card';
        case 'member':
            return 'user';
        case 'project':
            return 'project-diagram';
        default:
            return 'info-circle';
    }
}

// Helper function to format activity time
function formatActivityTime(timestamp) {
    if (!timestamp) return 'Unknown time';
    
    try {
        const date = new Date(timestamp);
        if (isNaN(date.getTime())) return 'Invalid date';
        
        // Format as "X minutes/hours/days ago" or "MMM DD, YYYY HH:MM"
        const now = new Date();
        const diffMs = now - date;
        const diffMinutes = Math.floor(diffMs / 60000);
        const diffHours = Math.floor(diffMs / 3600000);
        const diffDays = Math.floor(diffMs / 86400000);
        
        if (diffMinutes < 1) {
            return 'Just now';
        } else if (diffMinutes < 60) {
            return `${diffMinutes} minute${diffMinutes !== 1 ? 's' : ''} ago`;
        } else if (diffHours < 24) {
            return `${diffHours} hour${diffHours !== 1 ? 's' : ''} ago`;
        } else if (diffDays < 7) {
            return `${diffDays} day${diffDays !== 1 ? 's' : ''} ago`;
        } else {
            return date.toLocaleString();
        }
    } catch (e) {
        return 'Unknown time';
    }
}

// Function to show error message to user
function showErrorMessage(message) {
    // Remove existing error message if any
    const existingError = document.querySelector('.dashboard-error');
    if (existingError) {
        existingError.remove();
    }
    
    // Create error message element
    const errorElement = document.createElement('div');
    errorElement.className = 'dashboard-error';
    errorElement.textContent = message;
    errorElement.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        padding: 15px 20px;
        border-radius: 5px;
        color: white;
        background-color: #F44336;
        box-shadow: 0 4px 8px rgba(0,0,0,0.2);
        z-index: 10000;
        font-family: 'Inter', sans-serif;
        transform: translateX(0);
        transition: transform 0.3s ease;
    `;
    
    document.body.appendChild(errorElement);
    
    // Remove the error message after 5 seconds
    setTimeout(() => {
        errorElement.style.transform = 'translateX(120%)';
        setTimeout(() => {
            if (errorElement.parentNode) {
                errorElement.parentNode.removeChild(errorElement);
            }
        }, 300);
    }, 5000);
}

// Function to manually sync stats (newly added)
function syncAdminStats() {
    console.log('Manually syncing admin stats...');
    // Clear the cache to force a fresh data load
    dashboardDataCache.data = null;
    dashboardDataCache.timestamp = 0;
    loadAdminDashboardDataOptimized();
}

// Initialize dashboard data loading when DOM is ready
document.addEventListener('DOMContentLoaded', function() {
    console.log('DOM Content Loaded - Starting admin dashboard data loading');
    // Load admin dashboard data using optimized method
    loadAdminDashboardDataOptimized();
    
    // Set up periodic refresh for real-time updates (every 5 minutes)
    setInterval(loadAdminDashboardDataOptimized, 300000);
});

// Export functions for testing
window.loadAdminDashboardData = loadAdminDashboardData;
window.updateAdminDashboardCards = updateAdminDashboardCards;
window.syncAdminStats = syncAdminStats;