<?php
// MFT GROUP Initialize Default Settings via Web Interface
// This script initializes default system settings when accessed via browser

// Include database configuration
require_once 'database/config.php';
require_once 'api/settings.php';

// Only allow access via browser (not command line)
if (php_sapi_name() === 'cli') {
    echo "This script should be run via web browser, not command line.\n";
    exit(1);
}

// Set content type
header("Content-Type: text/html; charset=UTF-8");

// Default settings to insert
$default_settings = [
    // General Settings
    'orgName' => 'MFT GROUP',
    'orgAcronym' => 'MFT',
    'orgDescription' => 'Empowering our community through collective savings, mutual support, and sustainable financial growth since 2013.',
    'orgEmail' => 'info@mftselfhelpgroup.co.ke',
    'orgPhone' => '+254 720 327 807',
    'orgAddress' => 'Nairobi, Kenya',
    'currency' => 'KES',
    'timezone' => 'EAT',
    
    // Organization Settings
    'orgType' => 'chama',
    'chairperson' => 'John Mwangi',
    'secretary' => 'Mary Wanjiru',
    'treasurer' => 'Peter Kamau',
    'membersCount' => 38,
    'foundedDate' => '2013-05-15',
    
    // Membership Settings
    'membershipStatus' => 'open',
    'minContribution' => 500,
    'maxLoanMultiplier' => 3,
    'membershipFee' => 1000,
    'lateFee' => 100,
    'requireApproval' => true,
    'allowGuests' => false,
    'approvalType' => 'manual',
    'maxMembers' => 100,
    'renewalPeriod' => 365,
    'pendingStatus' => true,
    'activeStatus' => true,
    'suspendedStatus' => true,
    'terminatedStatus' => true,
    'gracePeriod' => 30,
    'membershipTiers' => 'single',
    'tierBenefits' => 'Regular members have standard voting rights and access to all group services.',
    
    // Loan Settings
    'loanStatus' => 'enabled',
    'interestRate' => 12,
    'savingsInterest' => 8,
    'dividendPercentage' => 15,
    'fineRate' => 5,
    'maxLoanTerm' => 12,
    'enableDividends' => true,
    'compoundInterest' => false,
    'maxLoanAmount' => 50000,
    'loanTerms' => '3,6,9,12',
    'lateInterest' => 15,
    'gracePeriodDays' => 7,
    'collateralPolicy' => 'required',
    'minSavings' => 5000,
    'overdue1Day' => true,
    'overdue3Days' => true,
    'overdue7Days' => true,
    'overdue14Days' => true,
    'savingsProducts' => 'regular',
    'savingsTypes' => 'Regular Savings,Emergency Fund,Fixed Deposit',
    'savingsInterestRates' => '8,6,10',
    'lockInPeriods' => '0,3,12',
    'minWithdrawal' => 1000,
    'withdrawalFrequency' => 'anytime',
    'scheduledContributions' => true,
    
    // Notification Settings
    'notificationEmail' => 'notifications@mftselfhelpgroup.co.ke',
    'emailMember' => true,
    'smsMember' => false,
    'pushMember' => false,
    'emailAdmin' => true,
    'smsAdmin' => false,
    'pushAdmin' => true,
    'contributionReminder' => '1day',
    'meetingReminder' => '3days',
    
    // Security Settings
    'sessionTimeout' => 30,
    'passwordExpiry' => 90,
    'ipWhitelist' => "192.168.1.0/24\n203.0.113.42",
    'loginAttempts' => 5,
    
    // Content Settings
    'aboutPage' => true,
    'projectsPage' => true,
    'galleryPage' => true,
    'downloadsPage' => true,
    'bankingPage' => true,
    'menuOrder' => 'Home,About Us,Projects,Membership,Gallery,Downloads,Banking Details,Contact Us',
    'maxGalleryImages' => 50,
    'maxFileSize' => 5,
    'allowedFileTypes' => 'pdf,doc,docx,xls,xlsx,jpg,png',
    'maxDownloadSize' => 10,
    
    // Officials Settings
    'officialsDisplay' => 'all',
    'officialsOrder' => 'Chairperson,Secretary,Treasurer,Vice Chairperson,Committee Members',
    'officialRoles' => 'Chairperson,Secretary,Treasurer,Vice Chairperson,Committee Member',
    'termLength' => 24,
    'termLimits' => true,
    'transitionPeriod' => 30,
    'electionNotice' => true,
    
    // Banking Settings
    'defaultCurrency' => 'KES',
    'primaryAccount' => 'account1',
    'accountVisibility' => 'members',
    'enableReconciliation' => true,
    'reconciliationThreshold' => 1000,
    
    // Documentation Settings
    'docStorageLocation' => 'local',
    'storageQuota' => 1000,
    'enableVersioning' => true,
    'maxVersions' => 10,
    'archivePeriod' => 12,
    'deletePeriod' => 60,
    
    // Roles Settings
    'adminRoles' => 'Super Admin,Financial Admin,Content Admin,Viewer',
    'defaultRole' => 'super',
    'manageLoans' => true,
    'manageMembers' => true,
    'manageContent' => true,
    'viewReports' => true,
    'disableAccounts' => true,
    'deleteAccounts' => false,
    
    // System Settings
    'backupFrequency' => 'daily',
    'backupRetention' => 30,
    'maintenanceMode' => false,
    'maintenanceMessage' => 'The system is currently under maintenance. We\'ll be back shortly.',
    'auditTrail' => true,
    'logRetention' => 90,
    
    // Integration Settings
    'mpesaStatus' => 'enabled',
    'mpesaKey' => 'sk_live_xxxxxxxxxxxxxxxx',
    'mpesaSecret' => 'sk_live_xxxxxxxxxxxxxxxx',
    'smsProvider' => 'twilio',
    'smsApiKey' => 'sk_live_xxxxxxxxxxxxxxxx',
    'smsSender' => 'MFTGROUP',
    'smtpHost' => 'smtp.gmail.com',
    'smtpPort' => 587,
    'smtpUser' => 'notifications@mftselfhelpgroup.co.ke',
    'smtpPass' => 'xxxxxxxxxxxxxxxx',
    
    // Branding Settings
    'primaryColor' => '#FF5C1B',
    'secondaryColor' => '#001a33',
    'themeMode' => 'dark',
    'fontFamily' => 'inter'
];

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Initialize Default Settings - MFT GROUP</title>
    <style>
        body {
            font-family: 'Inter', sans-serif;
            background: #1a1a1a;
            color: #fff;
            max-width: 800px;
            margin: 0 auto;
            padding: 20px;
        }
        .container {
            background: #2a2a2a;
            border-radius: 10px;
            padding: 30px;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.3);
        }
        h1 {
            color: #FF5C1B;
            text-align: center;
        }
        .settings-list {
            max-height: 400px;
            overflow-y: auto;
            background: #333;
            padding: 15px;
            border-radius: 5px;
            margin: 20px 0;
        }
        .setting-item {
            padding: 8px 0;
            border-bottom: 1px solid #444;
        }
        .setting-item:last-child {
            border-bottom: none;
        }
        .actions {
            text-align: center;
            margin: 20px 0;
        }
        button {
            background: linear-gradient(135deg, #FF5C1B, #e05015);
            color: white;
            border: none;
            padding: 12px 24px;
            border-radius: 5px;
            font-size: 16px;
            cursor: pointer;
            margin: 0 10px;
        }
        button:hover {
            background: linear-gradient(135deg, #e05015, #c2400f);
        }
        .result {
            padding: 15px;
            border-radius: 5px;
            margin: 20px 0;
            display: none;
        }
        .success {
            background: #28a745;
            display: block;
        }
        .error {
            background: #dc3545;
            display: block;
        }
        .status {
            text-align: center;
            font-weight: bold;
            margin: 10px 0;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>Initialize Default Settings</h1>
        <p>This script will initialize the default system settings for the MFT GROUP admin panel.</p>
        
        <h2>Settings to be initialized (<?php echo count($default_settings); ?> total):</h2>
        <div class="settings-list">
            <?php foreach ($default_settings as $key => $value): ?>
                <div class="setting-item">
                    <strong><?php echo htmlspecialchars($key); ?></strong>: 
                    <?php 
                    if (is_bool($value)) {
                        echo $value ? 'true' : 'false';
                    } elseif (is_array($value)) {
                        echo htmlspecialchars(json_encode($value));
                    } else {
                        echo htmlspecialchars($value);
                    }
                    ?>
                </div>
            <?php endforeach; ?>
        </div>
        
        <div class="actions">
            <button id="initSettings">Initialize Settings</button>
            <button id="goToAdmin">Go to Admin Settings</button>
        </div>
        
        <div id="result" class="result"></div>
        <div id="status" class="status"></div>
    </div>

    <script>
        document.getElementById('initSettings').addEventListener('click', function() {
            const button = this;
            const originalText = button.innerHTML;
            button.innerHTML = 'Initializing...';
            button.disabled = true;
            
            const statusDiv = document.getElementById('status');
            const resultDiv = document.getElementById('result');
            
            // Send request to initialize settings
            fetch('api/init-default-settings-api.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                }
            })
            .then(response => {
                // Handle non-JSON responses (like error pages)
                const contentType = response.headers.get('content-type');
                if (!contentType || !contentType.includes('application/json')) {
                    return response.text().then(text => {
                        throw new Error('Server error: ' + text);
                    });
                }
                return response.json();
            })
            .then(data => {
                if (data.status === 'success') {
                    resultDiv.className = 'result success';
                    resultDiv.innerHTML = 'Settings initialized successfully! ' + data.message;
                } else {
                    resultDiv.className = 'result error';
                    resultDiv.innerHTML = 'Error: ' + (data.error || 'Unknown error');
                }
                statusDiv.innerHTML = 'Completed';
            })
            .catch(error => {
                resultDiv.className = 'result error';
                resultDiv.innerHTML = 'Error: ' + error.message;
                statusDiv.innerHTML = 'Failed';
                console.error('Fetch error:', error);
            })
            .finally(() => {
                button.innerHTML = originalText;
                button.disabled = false;
            });
        });
        
        document.getElementById('goToAdmin').addEventListener('click', function() {
            window.location.href = 'admin-settings.html';
        });
    </script>
</body>
</html>