-- MFT GROUP Financial System Database Schema
-- Version: 1.0
-- Date: 2025-10-05

-- Drop existing tables if they exist (for development purposes)
-- In production, you would use proper migration scripts
DROP TABLE IF EXISTS audit_logs;
DROP TABLE IF EXISTS dividends;
DROP TABLE IF EXISTS loan_repayments;
DROP TABLE IF EXISTS loans;
DROP TABLE IF EXISTS contributions;
DROP TABLE IF EXISTS payments;
DROP TABLE IF EXISTS documents;
DROP TABLE IF EXISTS meetings;
DROP TABLE IF EXISTS member_roles;
DROP TABLE IF EXISTS roles;
DROP TABLE IF EXISTS members;
DROP TABLE IF EXISTS accounts;
DROP TABLE IF EXISTS account_types;
DROP TABLE IF EXISTS system_settings;
DROP TABLE IF EXISTS member_projects;
DROP TABLE IF EXISTS projects;

-- Create system settings table
CREATE TABLE system_settings (
    id INT PRIMARY KEY AUTO_INCREMENT,
    setting_key VARCHAR(100) UNIQUE NOT NULL,
    setting_value TEXT,
    description VARCHAR(255),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

-- Create account types table
CREATE TABLE account_types (
    id INT PRIMARY KEY AUTO_INCREMENT,
    name VARCHAR(50) UNIQUE NOT NULL,
    description TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Insert default account types
INSERT INTO account_types (name, description) VALUES
('Savings', 'Member savings account for regular contributions'),
('Loan', 'Loan account for borrowing'),
('Welfare', 'Welfare fund for emergencies'),
('Project', 'Project fund for group investments'),
('Dividend', 'Dividend distribution account'),
('Expense', 'General expense account');

-- Create accounts table
CREATE TABLE accounts (
    id INT PRIMARY KEY AUTO_INCREMENT,
    name VARCHAR(100) NOT NULL,
    account_type_id INT NOT NULL,
    balance DECIMAL(15, 2) DEFAULT 0.00,
    description TEXT,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (account_type_id) REFERENCES account_types(id)
);

-- Insert default accounts
INSERT INTO accounts (name, account_type_id, balance, description) VALUES
('Savings Account', 1, 0.00, 'Main savings account for member contributions'),
('Welfare Fund', 4, 0.00, 'Emergency and social welfare fund'),
('Project Fund', 4, 0.00, 'Fund for ongoing projects and investments'),
('Loan Account', 2, 0.00, 'Main loan account for lending operations'),
('Dividend Account', 5, 0.00, 'Account for dividend distributions'),
('Operating Expense', 6, 0.00, 'General operating expenses');

-- Create members table
CREATE TABLE members (
    id INT PRIMARY KEY AUTO_INCREMENT,
    member_id VARCHAR(20) UNIQUE NOT NULL,
    first_name VARCHAR(50) NOT NULL,
    last_name VARCHAR(50) NOT NULL,
    email VARCHAR(100) UNIQUE,
    phone VARCHAR(20),
    id_number VARCHAR(20) UNIQUE,
    date_of_birth DATE,
    gender ENUM('Male', 'Female', 'Other'),
    address TEXT,
    city VARCHAR(50),
    county VARCHAR(50),
    postal_code VARCHAR(10),
    country VARCHAR(50) DEFAULT 'Kenya',
    registration_date DATE NOT NULL,
    status ENUM('Pending', 'Active', 'Inactive', 'Suspended') DEFAULT 'Pending',
    role VARCHAR(20) DEFAULT 'member',
    profile_picture VARCHAR(255),
    last_login TIMESTAMP NULL,
    last_login_ip VARCHAR(45) NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

-- Create roles table
CREATE TABLE roles (
    id INT PRIMARY KEY AUTO_INCREMENT,
    name VARCHAR(50) UNIQUE NOT NULL,
    description TEXT,
    permissions TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Insert default roles
INSERT INTO roles (name, description, permissions) VALUES
('Admin', 'System administrator with full access', 'all'),
('Treasurer', 'Financial manager with accounting access', 'finance,reports,contributions,loans'),
('Secretary', 'Member and document manager', 'members,documents,meetings'),
('Chairman', 'Chairman with leadership responsibilities', 'members,meetings,reports'),
('Assistant Chairman', 'Assistant Chairman supporting the Chairman', 'members,meetings,reports'),
('Assistant Secretary', 'Assistant Secretary supporting the Secretary', 'members,documents,meetings'),
('Loans Officer', 'Loans officer managing loan applications', 'loans,reports'),
('Member', 'Regular member with limited access', 'profile,contributions,loans,documents');

-- Create member_roles table (many-to-many relationship)
CREATE TABLE member_roles (
    id INT PRIMARY KEY AUTO_INCREMENT,
    member_id INT NOT NULL,
    role_id INT NOT NULL,
    assigned_by INT,
    assigned_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (member_id) REFERENCES members(id) ON DELETE CASCADE,
    FOREIGN KEY (role_id) REFERENCES roles(id),
    FOREIGN KEY (assigned_by) REFERENCES members(id),
    UNIQUE KEY unique_member_role (member_id, role_id)
);

-- Create meetings table
CREATE TABLE meetings (
    id INT PRIMARY KEY AUTO_INCREMENT,
    title VARCHAR(255) NOT NULL,
    description TEXT,
    meeting_date DATE NOT NULL,
    start_time TIME,
    end_time TIME,
    location VARCHAR(255),
    created_by INT NOT NULL,
    status ENUM('Scheduled', 'Completed', 'Cancelled') DEFAULT 'Scheduled',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (created_by) REFERENCES members(id)
);

-- Create meeting_attendance table
CREATE TABLE meeting_attendance (
    id INT PRIMARY KEY AUTO_INCREMENT,
    meeting_id INT NOT NULL,
    member_id INT NOT NULL,
    attendance_status ENUM('Present', 'Absent', 'Late') NOT NULL DEFAULT 'Absent',
    reason TEXT,
    fine_imposed DECIMAL(10, 2) DEFAULT 0.00,
    fine_status ENUM('Pending', 'Paid', 'Waived') DEFAULT 'Pending',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (meeting_id) REFERENCES meetings(id) ON DELETE CASCADE,
    FOREIGN KEY (member_id) REFERENCES members(id) ON DELETE CASCADE,
    UNIQUE KEY unique_meeting_member (meeting_id, member_id)
);

-- Create documents table
CREATE TABLE documents (
    id INT PRIMARY KEY AUTO_INCREMENT,
    title VARCHAR(200) NOT NULL,
    description TEXT,
    file_path VARCHAR(500),
    file_type VARCHAR(50),
    file_size INT,
    category ENUM('KYC', 'Agreement', 'Receipt', 'Report', 'Meeting', 'Other') DEFAULT 'Other',
    uploaded_by INT NOT NULL,
    related_member_id INT,
    related_meeting_id INT,
    is_public BOOLEAN DEFAULT FALSE,
    status ENUM('pending', 'available', 'processing') DEFAULT 'pending',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (uploaded_by) REFERENCES members(id),
    FOREIGN KEY (related_member_id) REFERENCES members(id),
    FOREIGN KEY (related_meeting_id) REFERENCES meetings(id)
);

-- Create payments table
CREATE TABLE payments (
    id INT PRIMARY KEY AUTO_INCREMENT,
    reference VARCHAR(50) UNIQUE NOT NULL,
    member_id INT,
    account_id INT NOT NULL,
    amount DECIMAL(15, 2) NOT NULL,
    payment_method ENUM('M-PESA', 'Bank Transfer', 'Cash', 'Cheque', 'Other') NOT NULL,
    transaction_id VARCHAR(100),
    description TEXT,
    status ENUM('Pending', 'Completed', 'Failed', 'Cancelled') DEFAULT 'Pending',
    processed_by INT,
    processed_at TIMESTAMP NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (member_id) REFERENCES members(id),
    FOREIGN KEY (account_id) REFERENCES accounts(id),
    FOREIGN KEY (processed_by) REFERENCES members(id)
);

-- Create contributions table
CREATE TABLE contributions (
    id INT PRIMARY KEY AUTO_INCREMENT,
    member_id INT NOT NULL,
    payment_id INT UNIQUE,
    amount DECIMAL(15, 2) NOT NULL,
    contribution_date DATE NOT NULL,
    period_month INT,
    period_year INT,
    description TEXT,
    status ENUM('Pending', 'Confirmed', 'Rejected') DEFAULT 'Pending',
    confirmed_by INT,
    confirmed_at TIMESTAMP NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (member_id) REFERENCES members(id),
    FOREIGN KEY (payment_id) REFERENCES payments(id),
    FOREIGN KEY (confirmed_by) REFERENCES members(id)
);

-- Create loans table
CREATE TABLE loans (
    id INT PRIMARY KEY AUTO_INCREMENT,
    reference VARCHAR(50) UNIQUE NOT NULL,
    member_id INT NOT NULL,
    loan_type ENUM('Emergency', 'Business', 'Education', 'Agriculture', 'Personal') NOT NULL,
    principal_amount DECIMAL(15, 2) NOT NULL,
    interest_rate DECIMAL(5, 2) NOT NULL, -- percentage
    total_amount DECIMAL(15, 2) NOT NULL,
    repayment_period INT NOT NULL, -- in months
    monthly_payment DECIMAL(15, 2) NOT NULL,
    purpose TEXT,
    application_date DATE NOT NULL,
    approval_date DATE NULL,
    disbursement_date DATE NULL,
    first_payment_date DATE NULL,
    status ENUM('Pending', 'Approved', 'Rejected', 'Active', 'Completed', 'Defaulted') DEFAULT 'Pending',
    approved_by INT,
    reviewer_notes TEXT,
    guarantor_id INT NULL,
    guarantor_approval_status ENUM('Pending', 'Approved', 'Rejected') DEFAULT 'Pending',
    guarantor_approval_date DATE NULL,
    is_topup_loan BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (member_id) REFERENCES members(id),
    FOREIGN KEY (approved_by) REFERENCES members(id),
    FOREIGN KEY (guarantor_id) REFERENCES members(id)
);

-- Create loan_repayments table
CREATE TABLE loan_repayments (
    id INT PRIMARY KEY AUTO_INCREMENT,
    loan_id INT NOT NULL,
    payment_id INT UNIQUE,
    amount DECIMAL(15, 2) NOT NULL,
    principal_amount DECIMAL(15, 2) NOT NULL,
    interest_amount DECIMAL(15, 2) NOT NULL,
    repayment_date DATE NOT NULL,
    status ENUM('Pending', 'Completed', 'Failed') DEFAULT 'Pending',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (loan_id) REFERENCES loans(id),
    FOREIGN KEY (payment_id) REFERENCES payments(id)
);

-- Create dividends table
CREATE TABLE dividends (
    id INT PRIMARY KEY AUTO_INCREMENT,
    member_id INT NOT NULL,
    amount DECIMAL(15, 2) NOT NULL,
    financial_year VARCHAR(9) NOT NULL, -- e.g., "2024/2025"
    payment_date DATE,
    status ENUM('Pending', 'Paid', 'Cancelled') DEFAULT 'Pending',
    payment_reference VARCHAR(50),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (member_id) REFERENCES members(id)
);

-- Create fines table
CREATE TABLE fines (
    id INT PRIMARY KEY AUTO_INCREMENT,
    member_id INT NOT NULL,
    amount DECIMAL(15, 2) NOT NULL,
    reason ENUM('Lateness', 'Absentism', 'Absent with apology', 'Use of Phone when in meeting', 'Drunkenness', 'Other') NOT NULL,
    description TEXT,
    status ENUM('Pending', 'Paid', 'Cancelled', 'Waived', 'Unpaid') DEFAULT 'Pending',
    payment_reference VARCHAR(50),
    fine_date DATE NOT NULL,
    paid_date DATE NULL,
    created_by INT NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (member_id) REFERENCES members(id),
    FOREIGN KEY (created_by) REFERENCES members(id)
);

-- Create audit_logs table
CREATE TABLE audit_logs (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT,
    action VARCHAR(100) NOT NULL,
    table_name VARCHAR(50),
    record_id INT,
    old_values TEXT,
    new_values TEXT,
    ip_address VARCHAR(45),
    user_agent TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES members(id)
);

-- Create projects table
CREATE TABLE projects (
    id INT PRIMARY KEY AUTO_INCREMENT,
    name VARCHAR(200) NOT NULL,
    description TEXT,
    location VARCHAR(255),
    completion_date DATE,
    status ENUM('Planning', 'Active', 'Completed', 'Cancelled') DEFAULT 'Planning',
    total_shares INT DEFAULT 0,
    share_price DECIMAL(15, 2) DEFAULT 0.00,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

-- Create member_projects table (many-to-many relationship)
CREATE TABLE member_projects (
    id INT PRIMARY KEY AUTO_INCREMENT,
    member_id INT NOT NULL,
    project_id INT NOT NULL,
    shares_owned INT DEFAULT 1,
    amount_contributed DECIMAL(15, 2) DEFAULT 0.00,
    join_date DATE NOT NULL,
    status ENUM('Active', 'Completed', 'Withdrawn') DEFAULT 'Active',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (member_id) REFERENCES members(id) ON DELETE CASCADE,
    FOREIGN KEY (project_id) REFERENCES projects(id) ON DELETE CASCADE,
    UNIQUE KEY unique_member_project (member_id, project_id)
);

-- Create indexes for better performance
CREATE INDEX idx_members_member_id ON members(member_id);
CREATE INDEX idx_members_email ON members(email);
CREATE INDEX idx_members_status ON members(status);
CREATE INDEX idx_payments_reference ON payments(reference);
CREATE INDEX idx_payments_status ON payments(status);
CREATE INDEX idx_contributions_member_id ON contributions(member_id);
CREATE INDEX idx_contributions_date ON contributions(contribution_date);
CREATE INDEX idx_loans_reference ON loans(reference);
CREATE INDEX idx_loans_status ON loans(status);
CREATE INDEX idx_loans_member_id ON loans(member_id);
CREATE INDEX idx_repayments_loan_id ON loan_repayments(loan_id);
CREATE INDEX idx_repayments_date ON loan_repayments(repayment_date);
CREATE INDEX idx_dividends_member_id ON dividends(member_id);
CREATE INDEX idx_dividends_year ON dividends(financial_year);
CREATE INDEX idx_fines_member_id ON fines(member_id);
CREATE INDEX idx_fines_status ON fines(status);
CREATE INDEX idx_fines_date ON fines(fine_date);
CREATE INDEX idx_documents_category ON documents(category);
CREATE INDEX idx_meetings_date ON meetings(meeting_date);
CREATE INDEX idx_audit_logs_user_id ON audit_logs(user_id);
CREATE INDEX idx_audit_logs_action ON audit_logs(action);
CREATE INDEX idx_audit_logs_created_at ON audit_logs(created_at);
CREATE INDEX idx_projects_status ON projects(status);
CREATE INDEX idx_projects_completion_date ON projects(completion_date);
CREATE INDEX idx_member_projects_member_id ON member_projects(member_id);
CREATE INDEX idx_member_projects_project_id ON member_projects(project_id);
CREATE INDEX idx_member_projects_status ON member_projects(status);

-- Insert default admin user
INSERT INTO members (member_id, first_name, last_name, email, phone, id_number, registration_date, status) VALUES
('ADM-2025-001', 'System', 'Administrator', 'admin@mftselfhelpgroup.co.ke', '+254700000000', 'ADMIN001', CURDATE(), 'Active');

-- Assign admin role to default admin user
INSERT INTO member_roles (member_id, role_id, assigned_by) 
SELECT m.id, r.id, m.id 
FROM members m, roles r 
WHERE m.member_id = 'ADM-2025-001' AND r.name = 'Admin';

-- Comment out sample data for production
/*
INSERT INTO members (member_id, first_name, last_name, email, phone, id_number, registration_date, status) VALUES
('MFT-2025-001', 'John', 'Doe', 'john.doe@example.com', '+254712345678', 'ID12345678', CURDATE(), 'Active'),
('MFT-2025-002', 'Jane', 'Smith', 'jane.smith@example.com', '+254723456789', 'ID23456789', CURDATE(), 'Active');

-- Insert sample account balances
UPDATE accounts SET balance = 45000.00 WHERE name = 'Savings Account';
UPDATE accounts SET balance = 10000.00 WHERE name = 'Welfare Fund';
UPDATE accounts SET balance = 20000.00 WHERE name = 'Project Fund';

-- Insert sample contributions
INSERT INTO payments (reference, account_id, amount, payment_method, status, processed_at) VALUES
('PAY-001', 1, 2500.00, 'M-PESA', 'Completed', NOW()),
('PAY-002', 1, 2500.00, 'M-PESA', 'Completed', NOW());

INSERT INTO contributions (member_id, payment_id, amount, contribution_date, period_month, period_year, status, confirmed_by, confirmed_at) VALUES
(2, 1, 2500.00, CURDATE(), MONTH(CURDATE()), YEAR(CURDATE()), 'Confirmed', 1, NOW()),
(3, 2, 2500.00, CURDATE(), MONTH(CURDATE()), YEAR(CURDATE()), 'Confirmed', 1, NOW());

-- Insert sample loan
INSERT INTO loans (reference, member_id, loan_type, principal_amount, interest_rate, total_amount, repayment_period, monthly_payment, purpose, application_date, status) VALUES
('LOAN-001', 2, 'Emergency', 10000.00, 12.00, 11200.00, 12, 933.33, 'Medical emergency', CURDATE(), 'Active');

-- Insert sample projects
INSERT INTO projects (name, description, location, completion_date, status, total_shares, share_price) VALUES
('Konza Land Purchase', '1 Acre land purchase in Konza for 8 members', 'Konza, Kenya', '2024-12-31', 'Active', 8, 50000.00),
('Bisil Land Purchase', '2 Acres land purchase in Bisil for 15 members', 'Bisil, Kenya', '2024-06-30', 'Completed', 15, 40000.00),
('Ndeiya Land Purchase', '50 x 100 plot in Ndeiya for 17 members', 'Ndeiya, Kenya', '2024-03-31', 'Completed', 17, 35000.00),
('Nairobi Office Complex', 'Office space in Westlands for 12 members', 'Westlands, Nairobi', '2025-03-31', 'Active', 12, 75000.00);

-- Insert sample member-project associations
INSERT INTO member_projects (member_id, project_id, shares_owned, amount_contributed, join_date, status) VALUES
(2, 1, 1, 50000.00, CURDATE(), 'Active'),
(2, 4, 1, 75000.00, CURDATE(), 'Active'),
(2, 2, 1, 40000.00, '2024-01-15', 'Completed'),
(2, 3, 1, 35000.00, '2023-10-20', 'Completed'),
(3, 1, 1, 50000.00, CURDATE(), 'Active'),
(3, 2, 1, 40000.00, '2024-01-15', 'Completed');
*/

-- Create missed_contributions table
CREATE TABLE missed_contributions (
    id INT PRIMARY KEY AUTO_INCREMENT,
    member_id INT NOT NULL,
    amount DECIMAL(15, 2) NOT NULL,
    missed_date DATE NOT NULL,
    reason TEXT,
    status ENUM('Unpaid', 'Paid', 'Waived') DEFAULT 'Unpaid',
    payment_date DATE NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (member_id) REFERENCES members(id) ON DELETE CASCADE
);

-- Create loan_types table
CREATE TABLE loan_types (
    id INT PRIMARY KEY AUTO_INCREMENT,
    name VARCHAR(100) NOT NULL,
    interest_rate DECIMAL(5,2) NOT NULL,
    max_amount DECIMAL(15,2) NULL,
    max_term INT NULL,
    description TEXT,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

-- Insert default loan types
INSERT INTO loan_types (name, interest_rate, max_amount, max_term, description) VALUES
('Emergency Loan', 12.00, 50000.00, 12, 'Short-term loan for emergencies'),
('Business Loan', 10.00, 200000.00, 24, 'Loan for business investments'),
('Education Loan', 8.00, 100000.00, 36, 'Loan for educational purposes'),
('Agriculture Loan', 9.00, 150000.00, 18, 'Loan for agricultural activities'),
('Personal Loan', 11.00, 75000.00, 24, 'General purpose personal loan');

-- Create admin_module_permissions table
CREATE TABLE admin_module_permissions (
    id INT PRIMARY KEY AUTO_INCREMENT,
    member_id INT NOT NULL,
    module_name VARCHAR(50) NOT NULL,
    access_level ENUM('none', 'view', 'full') DEFAULT 'none',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (member_id) REFERENCES members(id) ON DELETE CASCADE,
    UNIQUE KEY unique_member_module (member_id, module_name)
);

COMMIT;