<?php
// Database configuration for MFT GROUP Financial System
// This file contains the database connection settings
// For cPanel hosting, update the constants below with your cPanel database credentials

// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('log_errors', 1);

// Database configuration for XAMPP local development
// Update these values with your local database details
define('DB_HOST', 'localhost'); // Usually localhost on XAMPP
define('DB_NAME', 'mftgroup'); // Your local database name
define('DB_USER', 'root'); // Default XAMPP user
define('DB_PASS', ''); // Default XAMPP password (usually empty)
define('DB_CHARSET', 'utf8mb4');

// PDO connection options
$pdo_options = [
    PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
    PDO::ATTR_EMULATE_PREPARES   => false,
];

// Function to get database connection
function getDatabaseConnection() {
    static $pdo = null;
    
    if ($pdo === null) {
        $dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=" . DB_CHARSET;
        global $pdo_options;
        
        try {
            $pdo = new PDO($dsn, DB_USER, DB_PASS, $pdo_options);
            error_log("Database connection established successfully");
        } catch (PDOException $e) {
            error_log("Database connection failed: " . $e->getMessage());
            // For cPanel, you might want to display a user-friendly error
            die("Database connection error. Please check your configuration.");
        }
    }
    
    return $pdo;
}

// Function to execute a query and return results
function executeQuery($sql, $params = []) {
    try {
        $pdo = getDatabaseConnection();
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        return $stmt->fetchAll();
    } catch (PDOException $e) {
        error_log("Query execution failed: " . $e->getMessage());
        throw new Exception("Query execution failed: " . $e->getMessage());
    }
}

// Function to execute a non-query (INSERT, UPDATE, DELETE)
function executeNonQuery($sql, $params = []) {
    try {
        $pdo = getDatabaseConnection();
        $stmt = $pdo->prepare($sql);
        $result = $stmt->execute($params);
        
        // Log successful execution
        error_log("Query executed successfully: " . $sql . " with params: " . json_encode($params));
        
        return $stmt->rowCount();
    } catch (PDOException $e) {
        error_log("Non-query execution failed: " . $e->getMessage() . " SQL: " . $sql . " Params: " . json_encode($params));
        throw new Exception("Query execution failed: " . $e->getMessage());
    }
}

// Function to get a single row
function fetchSingleRow($sql, $params = []) {
    try {
        $pdo = getDatabaseConnection();
        $stmt = $pdo->prepare($sql);
        $result = $stmt->execute($params);
        
        // Log successful execution
        error_log("Query executed successfully: " . $sql . " with params: " . json_encode($params));
        
        $row = $stmt->fetch();
        return $row ? $row : null;
    } catch (PDOException $e) {
        error_log("Query execution failed: " . $e->getMessage() . " SQL: " . $sql . " Params: " . json_encode($params));
        throw new Exception("Query execution failed: " . $e->getMessage());
    }
}

// Function to begin transaction
function beginTransaction() {
    $pdo = getDatabaseConnection();
    $result = $pdo->beginTransaction();
    error_log("Transaction started: " . ($result ? "SUCCESS" : "FAILED"));
    return $result;
}

// Function to commit transaction
function commitTransaction() {
    $pdo = getDatabaseConnection();
    $in_transaction = $pdo->inTransaction();
    if ($in_transaction) {
        $result = $pdo->commit();
        error_log("Transaction committed: " . ($result ? "SUCCESS" : "FAILED"));
        return $result;
    } else {
        error_log("Attempted to commit transaction, but not in transaction");
        return false;
    }
}

// Function to rollback transaction
function rollbackTransaction() {
    $pdo = getDatabaseConnection();
    $in_transaction = $pdo->inTransaction();
    if ($in_transaction) {
        $result = $pdo->rollback();
        error_log("Transaction rolled back: " . ($result ? "SUCCESS" : "FAILED"));
        return $result;
    } else {
        error_log("Attempted to rollback transaction, but not in transaction");
        return false;
    }
}

// Function to execute an insert query and return the inserted ID
function executeInsert($sql, $params = []) {
    try {
        $pdo = getDatabaseConnection();
        $stmt = $pdo->prepare($sql);
        $result = $stmt->execute($params);
        
        // Log successful execution
        error_log("Insert query executed successfully: " . $sql . " with params: " . json_encode($params));
        
        // Return the last inserted ID
        return $pdo->lastInsertId();
    } catch (PDOException $e) {
        error_log("Insert query execution failed: " . $e->getMessage() . " SQL: " . $sql . " Params: " . json_encode($params));
        throw new Exception("Query execution failed: " . $e->getMessage());
    }
}

// Function to execute an update query
function executeUpdate($sql, $params = []) {
    return executeNonQuery($sql, $params);
}

// Function to execute a delete query
function executeDelete($sql, $params = []) {
    return executeNonQuery($sql, $params);
}

// Function to create an audit log entry with real-time notification
function createAuditLog($user_id, $action, $table_name, $record_id = null, $old_values = null, $new_values = null) {
    try {
        // Get client IP and user agent
        $ip_address = isset($_SERVER['REMOTE_ADDR']) ? $_SERVER['REMOTE_ADDR'] : '';
        $user_agent = isset($_SERVER['HTTP_USER_AGENT']) ? $_SERVER['HTTP_USER_AGENT'] : '';
        
        // Insert audit log into database
        $sql = "INSERT INTO audit_logs (user_id, action, table_name, record_id, old_values, new_values, ip_address, user_agent) 
                VALUES (?, ?, ?, ?, ?, ?, ?, ?)";
        $params = [$user_id, $action, $table_name, $record_id, 
                   $old_values ? json_encode($old_values) : null, 
                   $new_values ? json_encode($new_values) : null, 
                   $ip_address, $user_agent];
        
        $log_id = executeInsert($sql, $params);
        
        // Get the inserted audit log with user details
        $log_sql = "SELECT a.id, a.user_id, a.action, a.table_name, a.record_id, a.old_values, a.new_values,
                           a.ip_address, a.user_agent, a.created_at, m.first_name, m.last_name, m.member_id as member_ref
                    FROM audit_logs a
                    LEFT JOIN members m ON a.user_id = m.id
                    WHERE a.id = ?";
        $log = fetchSingleRow($log_sql, [$log_id]);
        
        // Send real-time notification
        // Note: We're not including the function here to avoid duplication
        // The function exists in audit-logs.php
        
        return $log_id;
    } catch (Exception $e) {
        error_log("Error creating audit log: " . $e->getMessage());
        return false;
    }
}

?>