// Authentication and logout functionality for MFT GROUP website
console.log('auth.js loaded successfully');

// Include API configuration
if (typeof getApiUrl === 'undefined') {
    // Load API config if not already loaded
    const script = document.createElement('script');
    script.src = 'js/api-config.js';
    document.head.appendChild(script);
}

// Check if user is logged in and update navigation buttons accordingly
function updateAuthButtons() {
    console.log('updateAuthButtons called');
    
    // Check if member is logged in
    const memberId = localStorage.getItem('memberId');
    console.log('memberId from localStorage:', memberId);
    
    // Check if admin is logged in (we'll use a simple flag for now)
    const isAdmin = localStorage.getItem('isAdmin') === 'true';
    console.log('isAdmin:', isAdmin);
    
    // Find all logout buttons
    const logoutBtns = document.querySelectorAll('#logoutBtn');
    console.log('Found logout buttons:', logoutBtns.length);
    
    // Check if we're on a dashboard page (member or admin)
    const isDashboardPage = isMemberDashboardPage() || isAdminDashboardPage();
    console.log('isDashboardPage:', isDashboardPage);
    
    // If we're on an admin dashboard page and isAdmin is not set, automatically set it
    if (isAdminDashboardPage() && !isAdmin) {
        console.log('On admin dashboard page but isAdmin not set, checking server session');
        checkAdminSession();
        return; // Return early as checkAdminSession() will call updateAuthButtons() again
    }
    
    // Update logout buttons based on authentication status and page type
    logoutBtns.forEach((btn, index) => {
        console.log(`Processing logout button ${index}:`, btn);
        if (memberId || isAdmin || isDashboardPage) {
            // User is logged in or on a dashboard page, ensure it's a logout button
            console.log('User is logged in, ensuring logout button');
            btn.textContent = 'Logout';
            // Remove href for logout button to prevent navigation
            if (btn.tagName === 'A') {
                btn.removeAttribute('href');
            }
            // Remove any existing event listeners by cloning the element
            const newBtn = btn.cloneNode(true);
            newBtn.addEventListener('click', function(e) {
                console.log('Logout button clicked');
                e.preventDefault();
                logout();
            });
            btn.parentNode.replaceChild(newBtn, btn);
        }
    });
    
    // Handle the special case for contributions management page
    const dashboardBtn = document.getElementById('dashboardBtn');
    let dashboardBtnConverted = false;
    if (dashboardBtn && isAdmin && isAdminDashboardPage()) {
        // Change the button to logout for admin on admin pages
        dashboardBtn.textContent = 'Logout';
        dashboardBtn.removeAttribute('href');
        // Remove any existing event listeners by cloning the element
        const newBtn = dashboardBtn.cloneNode(true);
        newBtn.addEventListener('click', function(e) {
            console.log('Logout button clicked');
            e.preventDefault();
            logout();
        });
        dashboardBtn.parentNode.replaceChild(newBtn, dashboardBtn);
        dashboardBtnConverted = true;
    } else if (dashboardBtn && !isAdmin && !memberId && isDashboardPage) {
        // Ensure it's a login button for non-logged in users
        dashboardBtn.textContent = 'Member Login';
        dashboardBtn.href = 'member-login.html';
    }
    
    // If user is logged in and on a dashboard page, add logout button if it doesn't exist
    // But only if we haven't already converted the dashboardBtn
    if ((memberId || isAdmin) && isDashboardPage && !dashboardBtnConverted) {
        // Check if logout button exists, if not add it
        if (document.querySelectorAll('#logoutBtn').length === 0) {
            // Try to find a suitable place to add the logout button
            const navButtons = document.querySelector('.nav-buttons');
            if (navButtons) {
                const logoutBtn = document.createElement('a');
                logoutBtn.id = 'logoutBtn';
                logoutBtn.className = 'btn btn-orange';
                logoutBtn.textContent = 'Logout';
                logoutBtn.href = '#'; // Prevent navigation
                logoutBtn.addEventListener('click', function(e) {
                    e.preventDefault();
                    logout();
                });
                navButtons.appendChild(logoutBtn);
            }
        }
    }
}

// Check if we're on a member dashboard page
function isMemberDashboardPage() {
    const memberPages = [
        'dashboard.html',
        'contributions.html',
        'loan-status.html',
        'fines.html',
        'member-projects.html',
        'documents.html',
        'payments.html'
    ];
    
    const currentPage = window.location.pathname.split('/').pop();
    return memberPages.includes(currentPage);
}

// Check if we're on an admin dashboard page
function isAdminDashboardPage() {
    const adminPages = [
        'admin-dashboard.html',
        'admin-loans.html',
        'member-management.html',
        'contributions-management.html',
        'loans-management.html',
        'finance-management.html',
        'audit-trail.html',
        'banking.html',
        'organization-details.html',
        'dividends.html',
        'admin-projects.html'
    ];
    
    const currentPage = window.location.pathname.split('/').pop();
    return adminPages.includes(currentPage);
}

// Check admin session status
function checkAdminSession() {
    console.log('Checking admin session status');
    
    // Make API call to check if admin is authenticated
    fetch('api/admin-auth.php', {
        method: 'GET',
        credentials: 'include' // Include cookies/sessions
    })
    .then(response => response.json())
    .then(data => {
        console.log('Admin session check result:', data);
        if (data.authenticated) {
            // Admin is authenticated, set the flag
            localStorage.setItem('isAdmin', 'true');
            console.log('Admin session verified, isAdmin flag set');
        } else {
            // Admin is not authenticated, clear the flag
            localStorage.removeItem('isAdmin');
            console.log('Admin session not verified, isAdmin flag removed');
        }
        // Update auth buttons after checking session
        updateAuthButtons();
    })
    .catch(error => {
        console.error('Error checking admin session:', error);
        // On error, assume not authenticated
        localStorage.removeItem('isAdmin');
        updateAuthButtons();
    });
}

// Logout function
function logout() {
    // Remove member ID from localStorage
    localStorage.removeItem('memberId');
    
    // Remove admin flag from localStorage
    localStorage.removeItem('isAdmin');
    
    // Make API call to destroy server session
    fetch('api/admin-auth.php', {
        method: 'GET',
        credentials: 'include'
    })
    .then(() => {
        console.log('Server session cleared');
    })
    .catch(error => {
        console.error('Error clearing server session:', error);
    })
    .finally(() => {
        // Redirect based on current page
        const currentPage = window.location.pathname.split('/').pop();
        
        // If we're on an admin page, redirect to home
        if (currentPage && isAdminDashboardPage()) {
            window.location.href = 'index.html';
        } else {
            // For member pages, redirect to home
            window.location.href = 'index.html';
        }
    });
}

// Open signin modal
function openSigninModal() {
    // Create modal if it doesn't exist
    let modal = document.getElementById('signinModal');
    if (!modal) {
        console.log('Creating new signin modal');
        modal = document.createElement('div');
        modal.id = 'signinModal';
        modal.className = 'modal';
        modal.innerHTML = `
            <div class="modal-content">
                <span class="close">&times;</span>
                <h2>Member Sign In</h2>
                <p>Enter your membership number to access your dashboard</p>
                <form id="signinForm">
                    <div class="form-group">
                        <label for="membershipNumber">Membership Number</label>
                        <input type="text" id="membershipNumber" placeholder="e.g. MFT12345678" required>
                    </div>
                    <button type="submit" class="btn btn-orange">Sign In</button>
                </form>
                <div id="signinError" class="error-message" style="display: none;"></div>
            </div>
        `;
        document.body.appendChild(modal);
        console.log('Modal created and added to DOM');
        
        // Add event listeners for close button
        const closeBtn = modal.querySelector('.close');
        console.log('Close button found:', closeBtn);
        if (closeBtn) {
            closeBtn.addEventListener('click', function(e) {
                console.log('Close button clicked');
                e.preventDefault();
                closeSigninModal();
            });
        } else {
            console.error('Close button not found in modal');
        }
        
        // Add event listener for clicking outside modal
        window.addEventListener('click', function(e) {
            console.log('Window clicked, target:', e.target);
            if (e.target === modal) {
                console.log('Clicked outside modal, closing');
                closeSigninModal();
            }
        });
        
        // Add event listener for form submission
        const signinForm = document.getElementById('signinForm');
        console.log('Signin form found:', signinForm);
        if (signinForm) {
            signinForm.addEventListener('submit', function(e) {
                console.log('Form submitted');
                e.preventDefault();
                handleSignin(e);
            });
        } else {
            console.error('Signin form not found');
        }
    } else {
        console.log('Using existing modal from HTML');
        // If modal already exists in HTML, make sure it's hidden initially
        modal.style.display = 'none';
        
        // Add event listeners for close button if not already added
        const closeBtn = modal.querySelector('.close');
        console.log('Close button found:', closeBtn);
        if (closeBtn && !closeBtn.hasAttribute('data-listener-added')) {
            closeBtn.addEventListener('click', function(e) {
                console.log('Close button clicked');
                e.preventDefault();
                closeSigninModal();
            });
            closeBtn.setAttribute('data-listener-added', 'true');
        }
        
        // Add event listener for form submission if not already added
        const signinForm = document.getElementById('signinForm');
        console.log('Signin form found:', signinForm);
        if (signinForm && !signinForm.hasAttribute('data-listener-added')) {
            signinForm.addEventListener('submit', function(e) {
                console.log('Form submitted');
                e.preventDefault();
                handleSignin(e);
            });
            signinForm.setAttribute('data-listener-added', 'true');
        }
    }
    
    // Show modal
    modal.style.display = 'block';
    
    // Focus on the membership number input
    const membershipInput = document.getElementById('membershipNumber');
    if (membershipInput) {
        membershipInput.focus();
    }
}

// Close signin modal
function closeSigninModal() {
    console.log('closeSigninModal called');
    const modal = document.getElementById('signinModal');
    if (modal) {
        console.log('Modal found, hiding it');
        modal.style.display = 'none';
        const signinError = document.getElementById('signinError');
        if (signinError) {
            signinError.style.display = 'none';
        }
    } else {
        console.log('Modal not found');
    }
}

// Handle signin form submission
function handleSignin(e) {
    e.preventDefault();
    const membershipNumber = document.getElementById('membershipNumber').value.trim();
    const signinError = document.getElementById('signinError');
    
    // Simple validation
    if (!membershipNumber) {
        signinError.textContent = 'Please enter your membership number';
        signinError.style.display = 'block';
        return;
    }
    
    // Check if it follows the MFT + ID Number format (case insensitive)
    const upperMembershipNumber = membershipNumber.toUpperCase();
    if (!upperMembershipNumber.startsWith('MFT') || upperMembershipNumber.length < 4) {
        signinError.textContent = 'Invalid membership number format. It should start with "MFT" followed by your ID number.';
        signinError.style.display = 'block';
        return;
    }
    
    // Use the uppercase version for verification
    const normalizedMembershipNumber = upperMembershipNumber;
    
    // Show loading state
    const submitButton = document.querySelector('#signinForm button[type="submit"]');
    const originalText = submitButton.textContent;
    submitButton.textContent = 'Authenticating...';
    submitButton.disabled = true;
    
    // Verify member exists and is approved
    verifyMemberExists(normalizedMembershipNumber)
        .then(result => {
            if (result === true) {
                try {
                    // Store membership number in localStorage for use on dashboard
                    localStorage.setItem('memberId', normalizedMembershipNumber);
                    
                    // Close modal
                    closeSigninModal();
                    
                    // Redirect to dashboard
                    window.location.href = 'dashboard.html';
                } catch (error) {
                    console.error('Error storing membership ID:', error);
                    signinError.textContent = 'An error occurred. Please try again.';
                    signinError.style.display = 'block';
                }
            } else if (result === 'pending') {
                signinError.textContent = 'Your membership application is pending approval. Please contact an administrator for approval before accessing the dashboard.';
                signinError.style.display = 'block';
            } else {
                signinError.textContent = 'Membership number not found or not approved. Please check your number or contact admin.';
                signinError.style.display = 'block';
            }
        })
        .catch(error => {
            console.error('Authentication error:', error);
            signinError.textContent = 'An error occurred during authentication. Please try again.';
            signinError.style.display = 'block';
        })
        .finally(() => {
            // Restore button state
            submitButton.textContent = originalText;
            submitButton.disabled = false;
        });
}

// API call to verify member exists and is approved
function verifyMemberExists(memberId) {
    return new Promise((resolve, reject) => {
        // Make API call to check if member exists and is approved
        console.log('Verifying member with ID:', memberId);
        // Use the API configuration to construct the URL
        const apiUrl = getApiUrl('members.php?member_id=' + encodeURIComponent(memberId));
        console.log('API URL:', apiUrl);
        
        fetch(apiUrl)
            .then(response => {
                console.log('API response status:', response.status);
                if (!response.ok) {
                    // If member not found (404), resolve as false
                    if (response.status === 404) {
                        console.log('Member not found (404)');
                        resolve(false);
                        return;
                    }
                    throw new Error(`HTTP error! status: ${response.status}`);
                }
                return response.json();
            })
            .then(data => {
                console.log('API response data:', data);
                // Check if member exists and is approved (Active status)
                if (data && data.status === 'Active') {
                    console.log('Member is active');
                    resolve(true);
                } else if (data && data.status === 'Pending') {
                    // Member exists but is pending approval
                    console.log('Member is pending approval');
                    resolve('pending');
                } else {
                    // Member exists but is not approved (Inactive or Suspended)
                    console.log('Member is not active');
                    resolve(false);
                }
            })
            .catch(error => {
                console.error('Error verifying member:', error);
                // For network errors, we should be more lenient to prevent locking users out
                // Resolve as true to allow access, but log the error
                console.log('Network error during member verification, allowing access');
                resolve(true);
            });
    });
}

// Initialize authentication functionality when DOM is loaded
document.addEventListener('DOMContentLoaded', function() {
    updateAuthButtons();
});

// Export functions for admin use
window.updateAuthButtons = updateAuthButtons;
window.logout = logout;

// Function to set admin login status
function setAdminLogin() {
    localStorage.setItem('isAdmin', 'true');
    updateAuthButtons();
}

// Export admin login function
window.setAdminLogin = setAdminLogin;