<?php
// MFT GROUP WebSocket Server
// This script provides real-time updates using WebSocket technology

// Note: This is a simplified example. In production, you would use a proper WebSocket server
// like Ratchet (https://github.com/ratchetphp/Ratchet) or Swoole.

// For demonstration purposes, we'll create a simple long-polling endpoint
// that can be used as an alternative to WebSocket

// Enable CORS for frontend access - more permissive headers
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Credentials: true");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS");
header("Access-Control-Allow-Headers: Origin, X-Requested-With, Content-Type, Accept, Authorization");
header("Content-Type: application/json; charset=UTF-8");

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Include database configuration
require_once __DIR__ . '/../database/config.php';

// Create notifications table if it doesn't exist
createNotificationsTable();

$method = $_SERVER['REQUEST_METHOD'];

// Add debugging information
error_log("WebSocket server called with method: $method");

// Ensure we handle both HTTP and HTTPS requests properly
if (!isset($_SERVER['HTTPS']) || $_SERVER['HTTPS'] !== 'on') {
    // If not HTTPS, that's fine for localhost development
    error_log("WebSocket server accessed via HTTP");
} else {
    error_log("WebSocket server accessed via HTTPS");
}

if ($method === 'GET') {
    // Long-polling endpoint for real-time updates
    handleLongPolling();
} else if ($method === 'POST') {
    // Endpoint to notify clients of updates
    handleNotification();
} else {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
}

function createNotificationsTable() {
    try {
        $pdo = getDatabaseConnection();
        
        // Check if table exists
        $checkSql = "SHOW TABLES LIKE 'notifications'";
        $result = $pdo->query($checkSql);
        
        if ($result->rowCount() == 0) {
            // Create notifications table
            $sql = "
            CREATE TABLE notifications (
                id INT AUTO_INCREMENT PRIMARY KEY,
                type VARCHAR(50) NOT NULL,
                data TEXT,
                target ENUM('admin', 'member', 'all') DEFAULT 'all',
                member_id INT NULL,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                delivered_at TIMESTAMP NULL,
                INDEX idx_target (target),
                INDEX idx_member (member_id),
                INDEX idx_created (created_at),
                INDEX idx_type (type)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
            
            $pdo->exec($sql);
            error_log("Notifications table created successfully");
        }
    } catch (Exception $e) {
        error_log("Error creating notifications table: " . $e->getMessage());
    }
}

function handleLongPolling() {
    // Get parameters
    $dashboard_type = isset($_GET['dashboard']) ? $_GET['dashboard'] : 'admin';
    $member_id = isset($_GET['member_id']) ? $_GET['member_id'] : null;
    $last_update = isset($_GET['last_update']) ? intval($_GET['last_update']) : 0;
    $last_check = isset($_GET['last_check']) ? intval($_GET['last_check']) : 0;
    
    // Add debugging information
    error_log("Long polling request - dashboard: $dashboard_type, member_id: $member_id, last_update: $last_update");
    
    // Check for updates with a timeout to prevent hanging requests
    $start_time = time();
    $timeout = 15; // Reduce to 15 seconds to prevent hanging requests
    
    while (time() - $start_time < $timeout) {
        $has_updates = checkForUpdates($dashboard_type, $member_id, $last_update);
        
        if ($has_updates) {
            // Return updated data
            try {
                if ($dashboard_type === 'admin') {
                    $data = getAdminDashboardData();
                } else if ($dashboard_type === 'member_dashboard' && $member_id) {
                    $data = getMemberDashboardData($member_id);
                } else {
                    throw new Exception("Invalid dashboard type or missing member ID");
                }
                
                $data['timestamp'] = time();
                http_response_code(200);
                header('Content-Type: application/json');
                echo json_encode([
                    'status' => 'updated',
                    'data' => $data,
                    'timestamp' => time()
                ]);
                return;
            } catch (Exception $e) {
                error_log("Error getting dashboard data: " . $e->getMessage());
                http_response_code(500);
                header('Content-Type: application/json');
                echo json_encode(['error' => 'Failed to get dashboard data: ' . $e->getMessage()]);
                return;
            }
        }
        
        // Also check for notifications
        $notifications = getPendingNotifications($dashboard_type, $member_id, $last_check);
        if (!empty($notifications)) {
            http_response_code(200);
            header('Content-Type: application/json');
            echo json_encode([
                'status' => 'notifications',
                'notifications' => $notifications,
                'timestamp' => time()
            ]);
            return;
        }
        
        // Sleep for a short time to prevent excessive CPU usage
        usleep(500000); // 0.5 seconds
    }
    
    // No updates available within timeout
    http_response_code(200);
    header('Content-Type: application/json');
    echo json_encode([
        'status' => 'no_updates',
        'timestamp' => time()
    ]);
}

function handleNotification() {
    // In a real implementation, this would store the notification in a message queue
    // or database table that clients can check
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid JSON data']);
        return;
    }
    
    // Store notification in database
    try {
        $pdo = getDatabaseConnection();
        $type = isset($input['type']) ? $input['type'] : '';
        $data = isset($input['data']) ? json_encode($input['data']) : '{}';
        $target = isset($input['target']) ? $input['target'] : 'all';
        $member_id = isset($input['member_id']) ? $input['member_id'] : null;
        
        $sql = "INSERT INTO notifications (type, data, target, member_id) VALUES (?, ?, ?, ?)";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([$type, $data, $target, $member_id]);
        
        // Also send real-time update to the specific dashboard
        if ($target === 'member_dashboard' && $member_id) {
            // Get updated member dashboard data
            $dashboard_data = getMemberDashboardRealTimeData($member_id);
            
            // Add the notification type to the data
            $dashboard_data['type'] = $type;
            
            // Store the dashboard update notification
            $dashboard_sql = "INSERT INTO notifications (type, data, target, member_id) VALUES (?, ?, ?, ?)";
            $dashboard_stmt = $pdo->prepare($dashboard_sql);
            $dashboard_stmt->execute(['dashboard_update', json_encode($dashboard_data), 'member_dashboard', $member_id]);
        } else if ($target === 'admin') {
            // Get updated admin dashboard data
            $dashboard_data = getAdminDashboardRealTimeData();
            
            // Add the notification type to the data
            $dashboard_data['type'] = $type;
            
            // Store the dashboard update notification
            $dashboard_sql = "INSERT INTO notifications (type, data, target, member_id) VALUES (?, ?, ?, ?)";
            $dashboard_stmt = $pdo->prepare($dashboard_sql);
            $dashboard_stmt->execute(['dashboard_update', json_encode($dashboard_data), 'admin', null]);
        }
        
        http_response_code(200);
        echo json_encode([
            'status' => 'notification_received',
            'message' => 'Notification processed'
        ]);
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Failed to store notification: ' . $e->getMessage()]);
    }
}

function checkForUpdates($dashboard_type, $member_id, $last_update) {
    // In a real implementation, you would check a message queue or database
    // for notifications since the last update time
    
    // For this example, we'll check if there are any recent notifications
    try {
        $pdo = getDatabaseConnection();
        
        // Check for notifications newer than last_update
        $sql = "SELECT COUNT(*) as count FROM notifications 
                WHERE UNIX_TIMESTAMP(created_at) > ? 
                AND (target = ? OR target = 'all')";
        $params = [$last_update, $dashboard_type];
        
        if ($member_id && $dashboard_type === 'member_dashboard') {
            $sql .= " AND (member_id = ? OR member_id IS NULL)";
            $params[] = $member_id;
        }
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        $result = $stmt->fetch();
        
        return $result['count'] > 0;
    } catch (Exception $e) {
        error_log("Error checking for updates: " . $e->getMessage());
        return false;
    }
}

function getPendingNotifications($target, $member_id, $last_check) {
    try {
        $pdo = getDatabaseConnection();
        
        $sql = "SELECT id, type, data, created_at FROM notifications 
                WHERE (target = ? OR target = 'all') 
                AND UNIX_TIMESTAMP(created_at) > ?
                AND delivered_at IS NULL";
        $params = [$target, $last_check];
        
        // Fix the target check for member dashboard
        if ($member_id && $target === 'member_dashboard') {
            $sql .= " AND (member_id = ? OR member_id IS NULL)";
            $params[] = $member_id;
        }
        
        $sql .= " ORDER BY created_at ASC LIMIT 10";
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        $notifications = $stmt->fetchAll();
        
        // Mark notifications as delivered
        if (!empty($notifications)) {
            $ids = array_column($notifications, 'id');
            $placeholders = str_repeat('?,', count($ids) - 1) . '?';
            $updateSql = "UPDATE notifications SET delivered_at = NOW() WHERE id IN ($placeholders)";
            $updateStmt = $pdo->prepare($updateSql);
            $updateStmt->execute($ids);
        }
        
        // Decode JSON data and process specific notification types
        foreach ($notifications as &$notification) {
            $notification['data'] = json_decode($notification['data'], true);
            
            // Process contribution updates
            if ($notification['type'] === 'contribution_update' && $member_id) {
                $notification['data']['contributions'] = getMemberContributionsData($member_id);
                // Also update recent activities
                $notification['data']['recent_activities'] = getMemberRecentActivities($member_id);
            }
            
            // Process fine updates
            if ($notification['type'] === 'fine_update' && $member_id) {
                $notification['data']['fines'] = getMemberFinesData($member_id);
                // Also update recent activities
                $notification['data']['recent_activities'] = getMemberRecentActivities($member_id);
            }
            
            // Process loan updates
            if ($notification['type'] === 'loan_update' && $member_id) {
                $notification['data']['loans'] = getMemberLoansData($member_id);
                // Also update recent activities
                $notification['data']['recent_activities'] = getMemberRecentActivities($member_id);
            }
            
            // Process activity updates
            if ($notification['type'] === 'activity_update' && $member_id) {
                $notification['data']['recent_activities'] = getMemberRecentActivities($member_id);
            }
            
            // Process audit trail updates
            if ($notification['type'] === 'audit_trail_update') {
                // Audit trail updates don't need additional processing
                // The data is already complete
            }
        }
        
        return $notifications;
    } catch (Exception $e) {
        error_log("Error getting notifications: " . $e->getMessage());
        return [];
    }
}

function getAdminDashboardData() {
    // This is a simplified version of the dashboard data
    // In a real implementation, you would call the actual functions
    
    $data = [];
    
    try {
        // Get total members
        $member_sql = "SELECT COUNT(*) as total, 
                              SUM(CASE WHEN status = 'Active' THEN 1 ELSE 0 END) as active,
                              SUM(CASE WHEN status = 'Pending' THEN 1 ELSE 0 END) as pending
                       FROM members";
        $data['members'] = fetchSingleRow($member_sql);
        
        // Get total contributions this month
        $contribution_sql = "SELECT COALESCE(SUM(amount), 0) as total,
                                    COUNT(*) as count
                             FROM contributions 
                             WHERE YEAR(contribution_date) = YEAR(CURDATE()) 
                             AND MONTH(contribution_date) = MONTH(CURDATE())";
        $data['contributions_this_month'] = fetchSingleRow($contribution_sql);
        
        // Get recent contributions
        $data['recent_contributions'] = getRecentContributionsData(5);
        
        // Get pending loans
        $pending_loans_sql = "SELECT COUNT(*) as count,
                                     COALESCE(SUM(principal_amount), 0) as total_amount
                              FROM loans 
                              WHERE status = 'Pending'";
        $data['pending_loans'] = fetchSingleRow($pending_loans_sql);
        
        // Get active loans
        $active_loans_sql = "SELECT COUNT(*) as count,
                                    COALESCE(SUM(principal_amount), 0) as total_amount,
                                    COALESCE(SUM(principal_amount), 0) as total_with_interest
                             FROM loans 
                             WHERE status = 'Active'";
        $data['active_loans'] = fetchSingleRow($active_loans_sql);
        
        // Get account balances
        $accounts_sql = "SELECT a.name, a.balance, a.account_type_id
                         FROM accounts a
                         WHERE a.is_active = 1
                         ORDER BY a.id";
        $data['accounts'] = executeQuery($accounts_sql);
        
        // Get financial summary
        $financial_summary_sql = "SELECT 
                                         (SELECT COALESCE(SUM(amount), 0) FROM contributions WHERE status = 'Confirmed') as total_contributions,
                                         (SELECT COALESCE(SUM(amount), 0) FROM fines WHERE status = 'Paid') as total_paid_fines,
                                         (SELECT COALESCE(SUM(principal_amount), 0) FROM loans WHERE status = 'Active') as total_loans,
                                         (SELECT COALESCE(SUM(balance), 0) FROM accounts WHERE account_type_id = (SELECT id FROM account_types WHERE name = 'Expense')) as total_expenses";
        $financial_data = fetchSingleRow($financial_summary_sql);
        
        $data['financial_summary'] = [
            'total_contributions' => (float)($financial_data['total_contributions'] ?? 0),
            'total_paid_fines' => (float)($financial_data['total_paid_fines'] ?? 0),
            'total_loans' => (float)($financial_data['total_loans'] ?? 0),
            'total_expenses' => (float)($financial_data['total_expenses'] ?? 0),
            'bank_balance' => (float)(($financial_data['total_contributions'] ?? 0) + ($financial_data['total_paid_fines'] ?? 0)),
            'total_withdrawals' => (float)($financial_data['total_loans'] ?? 0),
            'net_balance' => (float)(($financial_data['total_contributions'] ?? 0) + ($financial_data['total_paid_fines'] ?? 0) - ($financial_data['total_loans'] ?? 0) - ($financial_data['total_expenses'] ?? 0))
        ];
        
        // Get pending fines
        $fines_sql = "SELECT COALESCE(SUM(amount), 0) as total_pending_fines,
                             COUNT(*) as pending_fines_count
                      FROM fines 
                      WHERE status IN ('Pending', 'Unpaid')";
        $fines_result = fetchSingleRow($fines_sql);
        // Ensure numeric values
        $data['fines'] = [
            'total_pending_fines' => (float)($fines_result['total_pending_fines'] ?? 0),
            'pending_fines_count' => (int)($fines_result['pending_fines_count'] ?? 0)
        ];
        
        // Get paid fines (for Bank Balance calculation)
        $paid_fines_sql = "SELECT COALESCE(SUM(amount), 0) as total_paid_fines
                          FROM fines 
                          WHERE status = 'Paid'";
        $paid_fines_result = fetchSingleRow($paid_fines_sql);
        // Ensure numeric values
        $data['paid_fines'] = [
            'total_paid_fines' => (float)($paid_fines_result['total_paid_fines'] ?? 0)
        ];
        
        // Get project statistics
        $projects_sql = "SELECT COUNT(*) as active,
                                SUM(CASE WHEN status = 'Completed' THEN 1 ELSE 0 END) as completed
                         FROM projects 
                         WHERE status IN ('Active', 'Completed')";
        $projects_result = fetchSingleRow($projects_sql);
        // Ensure numeric values
        $data['projects'] = [
            'active' => (int)($projects_result['active'] ?? 0),
            'completed' => (int)($projects_result['completed'] ?? 0)
        ];
        
    } catch (Exception $e) {
        error_log("Error getting admin dashboard data: " . $e->getMessage());
        $data['error'] = $e->getMessage();
    }
    
    return $data;
}

function getMemberDashboardData($member_id) {
    // This is a simplified version of the member dashboard data
    // In a real implementation, you would call the actual functions
    
    $data = [];
    
    try {
        // Get member info - FIXED: Use member_id (string) instead of numeric ID
        $member_sql = "SELECT first_name, last_name, member_id, status, registration_date
                       FROM members 
                       WHERE member_id = ?";
        $data['member_info'] = fetchSingleRow($member_sql, [$member_id]);
        
        if (!$data['member_info']) {
            throw new Exception('Member not found');
        }
        
        // Get member's numeric ID for other queries
        $member_check_sql = "SELECT id FROM members WHERE member_id = ?";
        $member_result = fetchSingleRow($member_check_sql, [$member_id]);
        if (!$member_result) {
            throw new Exception('Member not found');
        }
        $member_numeric_id = $member_result['id'];
        
        // Get member contributions balance
        $contributions_sql = "SELECT COALESCE(SUM(amount), 0) as total_balance
                              FROM contributions 
                              WHERE member_id = ? AND status = 'Confirmed'";
        $data['contributions'] = fetchSingleRow($contributions_sql, [$member_numeric_id]);
        
        // Get member outstanding loans
        $loans_sql = "SELECT COALESCE(SUM(principal_amount), 0) as outstanding_balance,
                             COUNT(*) as active_loans
                      FROM loans 
                      WHERE member_id = ? AND status = 'Active'";
        $data['loans'] = fetchSingleRow($loans_sql, [$member_numeric_id]);
        
        // Get member dividends
        $dividends_sql = "SELECT COALESCE(SUM(amount), 0) as total_dividends
                          FROM dividends 
                          WHERE member_id = ? AND status = 'Paid'";
        $data['dividends'] = fetchSingleRow($dividends_sql, [$member_numeric_id]);
        
        // Get member projects
        $projects_sql = "SELECT p.id, p.name, p.description, p.location, p.completion_date, p.status
                         FROM projects p
                         JOIN member_projects mp ON p.id = mp.project_id
                         WHERE mp.member_id = ?
                         ORDER BY p.completion_date DESC";
        $data['projects'] = executeQuery($projects_sql, [$member_numeric_id]);
        
        // Get member fines
        $fines_sql = "SELECT COALESCE(SUM(amount), 0) as total_fines
                      FROM fines 
                      WHERE member_id = ? AND status = 'Unpaid'";
        $data['fines'] = fetchSingleRow($fines_sql, [$member_numeric_id]);
        
    } catch (Exception $e) {
        error_log("Error getting member dashboard data: " . $e->getMessage());
        throw $e; // Re-throw the exception so it can be handled upstream
    }
    
    return $data;
}

// Function to get recent contributions for real-time updates
function getRecentContributionsData($limit = 5) {
    try {
        $sql = "SELECT c.*, m.first_name, m.last_name, m.member_id, p.reference as payment_reference 
                FROM contributions c
                JOIN members m ON c.member_id = m.id
                LEFT JOIN payments p ON c.payment_id = p.id
                ORDER BY c.contribution_date DESC 
                LIMIT ?";
        
        $results = executeQuery($sql, [$limit]);
        
        // Add debugging to see what data is being returned
        error_log("Recent contributions data: " . json_encode($results));
        
        // Ensure numeric values are properly formatted
        foreach ($results as &$contribution) {
            $contribution['amount'] = (float)($contribution['amount'] ?? 0);
            // Make sure all expected fields are present
            if (!isset($contribution['payment_method'])) {
                $contribution['payment_method'] = 'N/A';
            }
            if (!isset($contribution['category'])) {
                $contribution['category'] = 'Regular';
            }
            if (!isset($contribution['status'])) {
                $contribution['status'] = 'Confirmed';
            }
        }
        
        return $results;
    } catch (Exception $e) {
        error_log("Error getting recent contributions data: " . $e->getMessage());
        return [];
    }
}

// Function to get specific member data for real-time updates
function getMemberContributionsData($member_id) {
    try {
        // Get member's numeric ID
        $member_check_sql = "SELECT id FROM members WHERE member_id = ?";
        $member_result = fetchSingleRow($member_check_sql, [$member_id]);
        if (!$member_result) {
            throw new Exception('Member not found');
        }
        $member_numeric_id = $member_result['id'];
        
        // Get member contributions summary
        $summary_sql = "SELECT 
                          COALESCE(SUM(CASE WHEN status = 'Confirmed' THEN amount ELSE 0 END), 0) as total_balance,
                          COUNT(*) as total_records
                        FROM contributions 
                        WHERE member_id = ?";
        $summary = fetchSingleRow($summary_sql, [$member_numeric_id]);
        
        return $summary;
    } catch (Exception $e) {
        error_log("Error getting member contributions data: " . $e->getMessage());
        return null;
    }
}

// Function to get specific member fines data for real-time updates
function getMemberFinesData($member_id) {
    try {
        // Get member's numeric ID
        $member_check_sql = "SELECT id FROM members WHERE member_id = ?";
        $member_result = fetchSingleRow($member_check_sql, [$member_id]);
        if (!$member_result) {
            throw new Exception('Member not found');
        }
        $member_numeric_id = $member_result['id'];
        
        // Get member fines summary
        $summary_sql = "SELECT 
                          COALESCE(SUM(CASE WHEN status = 'Unpaid' THEN amount ELSE 0 END), 0) as total_fines,
                          COUNT(*) as total_records
                        FROM fines 
                        WHERE member_id = ?";
        $summary = fetchSingleRow($summary_sql, [$member_numeric_id]);
        
        return $summary;
    } catch (Exception $e) {
        error_log("Error getting member fines data: " . $e->getMessage());
        return null;
    }
}

// Function to get specific member loans data for real-time updates
function getMemberLoansData($member_id) {
    try {
        // Get member's numeric ID
        $member_check_sql = "SELECT id FROM members WHERE member_id = ?";
        $member_result = fetchSingleRow($member_check_sql, [$member_id]);
        if (!$member_result) {
            throw new Exception('Member not found');
        }
        $member_numeric_id = $member_result['id'];
        
        // Get member loans summary
        $summary_sql = "SELECT 
                          COALESCE(SUM(principal_amount), 0) as outstanding_balance,
                          COUNT(*) as active_loans
                        FROM loans 
                        WHERE member_id = ? AND status = 'Active'";
        $summary = fetchSingleRow($summary_sql, [$member_numeric_id]);
        
        return $summary;
    } catch (Exception $e) {
        error_log("Error getting member loans data: " . $e->getMessage());
        return null;
    }
}

// Function to get member dashboard data for real-time updates
function getMemberDashboardRealTimeData($member_id) {
    $data = [];
    
    try {
        // Get contributions data
        $data['contributions'] = getMemberContributionsData($member_id);
        
        // Get fines data
        $data['fines'] = getMemberFinesData($member_id);
        
        // Get loans data
        $data['loans'] = getMemberLoansData($member_id);
        
        // Get recent activities
        $data['recent_activities'] = getMemberRecentActivities($member_id);
        
        // Get dividends data
        // Get member's numeric ID
        $member_check_sql = "SELECT id FROM members WHERE member_id = ?";
        $member_result = fetchSingleRow($member_check_sql, [$member_id]);
        if (!$member_result) {
            throw new Exception('Member not found');
        }
        $member_numeric_id = $member_result['id'];
        
        $dividends_sql = "SELECT 
                            COALESCE(SUM(amount), 0) as total_dividends
                          FROM dividends 
                          WHERE member_id = ? AND status = 'Paid'";
        $data['dividends'] = fetchSingleRow($dividends_sql, [$member_numeric_id]);
        
    } catch (Exception $e) {
        error_log("Error getting member dashboard real-time data: " . $e->getMessage());
        $data['error'] = $e->getMessage();
    }
    
    return $data;
}

// Function to get admin dashboard data for real-time updates
function getAdminDashboardRealTimeData() {
    $data = [];
    
    try {
        // Get member statistics
        $member_sql = "SELECT COUNT(*) as total, 
                              SUM(CASE WHEN status = 'Active' THEN 1 ELSE 0 END) as active,
                              SUM(CASE WHEN status = 'Pending' THEN 1 ELSE 0 END) as pending
                       FROM members";
        $member_result = fetchSingleRow($member_sql);
        // Ensure numeric values
        $data['members'] = [
            'total' => (int)($member_result['total'] ?? 0),
            'active' => (int)($member_result['active'] ?? 0),
            'pending' => (int)($member_result['pending'] ?? 0)
        ];
        
        // Get contribution statistics
        $contributions_sql = "SELECT COALESCE(SUM(amount), 0) as total_balance,
                                     COALESCE(SUM(CASE WHEN YEAR(contribution_date) = YEAR(CURDATE()) AND MONTH(contribution_date) = MONTH(CURDATE()) THEN amount ELSE 0 END), 0) as this_month
                              FROM contributions 
                              WHERE status = 'Confirmed'";
        $contributions_result = fetchSingleRow($contributions_sql);
        // Ensure numeric values
        $data['contributions'] = [
            'total_balance' => (float)($contributions_result['total_balance'] ?? 0),
            'this_month' => (float)($contributions_result['this_month'] ?? 0)
        ];
        
        // Get recent contributions
        $data['recentContributions'] = getRecentContributionsData(5);
        
        // Get loan statistics
        $loans_sql = "SELECT COALESCE(SUM(CASE WHEN status = 'Active' THEN principal_amount ELSE 0 END), 0) as outstanding_balance,
                             SUM(CASE WHEN status = 'Active' THEN 1 ELSE 0 END) as active_loans,
                             SUM(CASE WHEN status = 'Pending' THEN 1 ELSE 0 END) as pending_applications
                      FROM loans 
                      WHERE status IN ('Active', 'Pending')";
        $loans_result = fetchSingleRow($loans_sql);
        // Ensure numeric values
        $data['loans'] = [
            'outstanding_balance' => (float)($loans_result['outstanding_balance'] ?? 0),
            'active_loans' => (int)($loans_result['active_loans'] ?? 0),
            'pending_applications' => (int)($loans_result['pending_applications'] ?? 0)
        ];
        
        // Get project statistics
        $projects_sql = "SELECT COUNT(*) as active,
                                SUM(CASE WHEN status = 'Completed' THEN 1 ELSE 0 END) as completed
                         FROM projects 
                         WHERE status IN ('Active', 'Completed')";
        $projects_result = fetchSingleRow($projects_sql);
        // Ensure numeric values
        $data['projects'] = [
            'active' => (int)($projects_result['active'] ?? 0),
            'completed' => (int)($projects_result['completed'] ?? 0)
        ];
        
        // Get pending fines
        $fines_sql = "SELECT COALESCE(SUM(amount), 0) as total_pending_fines,
                             COUNT(*) as pending_fines_count
                      FROM fines 
                      WHERE status IN ('Pending', 'Unpaid')";
        $fines_result = fetchSingleRow($fines_sql);
        // Ensure numeric values
        $data['fines'] = [
            'total_pending_fines' => (float)($fines_result['total_pending_fines'] ?? 0),
            'pending_fines_count' => (int)($fines_result['pending_fines_count'] ?? 0)
        ];
        
        // Get paid fines (for Bank Balance calculation)
        $paid_fines_sql = "SELECT COALESCE(SUM(amount), 0) as total_paid_fines
                          FROM fines 
                          WHERE status = 'Paid'";
        $paid_fines_result = fetchSingleRow($paid_fines_sql);
        // Ensure numeric values
        $data['paid_fines'] = [
            'total_paid_fines' => (float)($paid_fines_result['total_paid_fines'] ?? 0)
        ];
        
        // Calculate financial summary
        $financial_summary_sql = "SELECT 
                                         (SELECT COALESCE(SUM(amount), 0) FROM contributions WHERE status = 'Confirmed') as total_contributions,
                                         (SELECT COALESCE(SUM(amount), 0) FROM fines WHERE status = 'Paid') as total_paid_fines,
                                         (SELECT COALESCE(SUM(principal_amount), 0) FROM loans WHERE status = 'Active') as total_loans,
                                         (SELECT COALESCE(SUM(balance), 0) FROM accounts WHERE account_type_id = (SELECT id FROM account_types WHERE name = 'Expense')) as total_expenses";
        $financial_data = fetchSingleRow($financial_summary_sql);
        
        $data['financial_summary'] = [
            'total_contributions' => (float)($financial_data['total_contributions'] ?? 0),
            'total_paid_fines' => (float)($financial_data['total_paid_fines'] ?? 0),
            'total_loans' => (float)($financial_data['total_loans'] ?? 0),
            'total_expenses' => (float)($financial_data['total_expenses'] ?? 0),
            'bank_balance' => (float)(($financial_data['total_contributions'] ?? 0) + ($financial_data['total_paid_fines'] ?? 0)),
            'total_withdrawals' => (float)($financial_data['total_loans'] ?? 0),
            'net_balance' => (float)(($financial_data['total_contributions'] ?? 0) + ($financial_data['total_paid_fines'] ?? 0) - ($financial_data['total_loans'] ?? 0) - ($financial_data['total_expenses'] ?? 0))
        ];
        
    } catch (Exception $e) {
        error_log("Error getting admin dashboard real-time data: " . $e->getMessage());
        $data['error'] = $e->getMessage();
    }
    
    return $data;
}
?>