<?php
// MFT GROUP Reports API
// This API provides reporting data for the admin dashboard

// Enable CORS for frontend access
header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Methods: GET, POST");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");

// Include database configuration
require_once __DIR__ . '/../database/config.php';

// Get request method
$method = $_SERVER['REQUEST_METHOD'];

try {
    switch ($method) {
        case 'GET':
            handleGetRequest();
            break;
        case 'POST':
            handlePostRequest();
            break;
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            break;
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Internal server error: ' . $e->getMessage()]);
}

function handleGetRequest() {
    if (isset($_GET['type'])) {
        switch ($_GET['type']) {
            case 'member':
                getMemberReport();
                break;
            case 'financial':
                getFinancialReport();
                break;
            case 'periodic':
                getPeriodicReport();
                break;
            case 'analytics':
                getAnalyticsData();
                break;
            default:
                http_response_code(400);
                echo json_encode(['error' => 'Invalid report type']);
        }
    } else {
        http_response_code(400);
        echo json_encode(['error' => 'Report type is required']);
    }
}

function handlePostRequest() {
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid JSON data']);
        return;
    }
    
    // Handle export requests
    if (isset($input['action']) && $input['action'] === 'export') {
        handleExportRequest($input);
    } else {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid action']);
    }
}

// Get member report data
function getMemberReport() {
    $member_id = isset($_GET['member_id']) ? $_GET['member_id'] : null;
    $date_range = isset($_GET['date_range']) ? $_GET['date_range'] : null;
    
    if (!$member_id) {
        http_response_code(400);
        echo json_encode(['error' => 'Member ID is required']);
        return;
    }
    
    try {
        // Build date range conditions for contributions
        $contribution_date_conditions = "";
        $contribution_params = [$member_id];
        
        // Build date range conditions for fines
        $fine_date_conditions = "";
        $fine_params = [$member_id];
        
        // Build date range conditions for dividends
        $dividend_date_conditions = "";
        $dividend_params = [$member_id];
        
        if ($date_range && $date_range !== 'all') {
            switch ($date_range) {
                case '30':
                    $contribution_date_conditions = " AND contribution_date >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)";
                    $fine_date_conditions = " AND fine_date >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)";
                    $dividend_date_conditions = " AND payment_date >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)";
                    break;
                case '90':
                    $contribution_date_conditions = " AND contribution_date >= DATE_SUB(CURDATE(), INTERVAL 90 DAY)";
                    $fine_date_conditions = " AND fine_date >= DATE_SUB(CURDATE(), INTERVAL 90 DAY)";
                    $dividend_date_conditions = " AND payment_date >= DATE_SUB(CURDATE(), INTERVAL 90 DAY)";
                    break;
                case '365':
                    $contribution_date_conditions = " AND contribution_date >= DATE_SUB(CURDATE(), INTERVAL 1 YEAR)";
                    $fine_date_conditions = " AND fine_date >= DATE_SUB(CURDATE(), INTERVAL 1 YEAR)";
                    $dividend_date_conditions = " AND payment_date >= DATE_SUB(CURDATE(), INTERVAL 1 YEAR)";
                    break;
                default:
                    // Custom date range
                    if (strpos($date_range, ',') !== false) {
                        list($from_date, $to_date) = explode(',', $date_range);
                        $contribution_date_conditions = " AND contribution_date BETWEEN ? AND ?";
                        $contribution_params[] = $from_date;
                        $contribution_params[] = $to_date;
                        
                        $fine_date_conditions = " AND fine_date BETWEEN ? AND ?";
                        $fine_params[] = $from_date;
                        $fine_params[] = $to_date;
                        
                        $dividend_date_conditions = " AND payment_date BETWEEN ? AND ?";
                        $dividend_params[] = $from_date;
                        $dividend_params[] = $to_date;
                    }
                    break;
            }
        }
        
        // Get member contributions with date filtering
        $contributions_sql = "SELECT amount, contribution_date, status, description as payment_method 
                             FROM contributions 
                             WHERE member_id = ? 
                             $contribution_date_conditions
                             ORDER BY contribution_date DESC";
        $contributions = executeQuery($contributions_sql, $contribution_params);
        
        // Get member loans (no date filtering for loans as they have different date fields)
        $loans_sql = "SELECT reference, principal_amount, interest_rate, total_amount, 
                             monthly_payment, first_payment_date, status 
                      FROM loans 
                      WHERE member_id = ? 
                      ORDER BY application_date DESC";
        $loans = executeQuery($loans_sql, [$member_id]);
        
        // Get member fines with date filtering
        $fines_sql = "SELECT amount, reason, fine_date, status 
                     FROM fines 
                     WHERE member_id = ? 
                     $fine_date_conditions
                     ORDER BY fine_date DESC";
        $fines = executeQuery($fines_sql, $fine_params);
        
        // Get member dividends with date filtering
        $dividends_sql = "SELECT amount, financial_year, payment_date, status 
                         FROM dividends 
                         WHERE member_id = ? 
                         $dividend_date_conditions
                         ORDER BY payment_date DESC";
        $dividends = executeQuery($dividends_sql, $dividend_params);
        
        http_response_code(200);
        echo json_encode([
            'contributions' => $contributions,
            'loans' => $loans,
            'fines' => $fines,
            'dividends' => $dividends
        ]);
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Error fetching member report data: ' . $e->getMessage()]);
    }
}

// Get financial report data
function getFinancialReport() {
    $period = isset($_GET['period']) ? $_GET['period'] : 'monthly';
    $year = isset($_GET['year']) ? $_GET['year'] : date('Y');
    
    try {
        // Get income data
        $income_data = getIncomeData($period, $year);
        
        // Get expense data
        $expense_data = getExpenseData($period, $year);
        
        // Get balance sheet data
        $balance_sheet_data = getBalanceSheetData();
        
        // Get bank reconciliation data
        $bank_reconciliation_data = getBankReconciliationData();
        
        http_response_code(200);
        echo json_encode([
            'income' => $income_data,
            'expenses' => $expense_data,
            'balance_sheet' => $balance_sheet_data,
            'bank_reconciliation' => $bank_reconciliation_data
        ]);
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Error fetching financial report data: ' . $e->getMessage()]);
    }
}

// Get periodic report data
function getPeriodicReport() {
    $report_type = isset($_GET['report_type']) ? $_GET['report_type'] : 'monthly-contribution';
    $period = isset($_GET['period']) ? $_GET['period'] : 'monthly';
    
    try {
        $data = [];
        
        switch ($report_type) {
            case 'monthly-contribution':
                $data = getMonthlyContributionData($period);
                break;
            case 'monthly-loan':
                $data = getMonthlyLoanData($period);
                break;
            case 'yearly-financial':
                $data = getYearlyFinancialData();
                break;
            case 'attendance':
                $data = getAttendanceData();
                break;
            default:
                $data = [];
        }
        
        http_response_code(200);
        echo json_encode([
            'report_type' => $report_type,
            'data' => $data
        ]);
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Error fetching periodic report data: ' . $e->getMessage()]);
    }
}

// Get analytics data
function getAnalyticsData() {
    $date_range = isset($_GET['date_range']) ? $_GET['date_range'] : '30';
    
    try {
        // Get contribution trends
        $contribution_trends = getContributionTrends($date_range);
        
        // Get top contributors
        $top_contributors = getTopContributors();
        
        // Get loan distribution
        $loan_distribution = getLoanDistribution();
        
        // Get income vs expense comparison
        $income_expense_comparison = getIncomeExpenseComparison();
        
        // Get member growth
        $member_growth = getMemberGrowth($date_range);
        
        // Get project cost breakdown
        $project_cost_breakdown = getProjectCostBreakdown();
        
        http_response_code(200);
        echo json_encode([
            'contribution_trends' => $contribution_trends,
            'top_contributors' => $top_contributors,
            'loan_distribution' => $loan_distribution,
            'income_expense_comparison' => $income_expense_comparison,
            'member_growth' => $member_growth,
            'project_cost_breakdown' => $project_cost_breakdown
        ]);
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Error fetching analytics data: ' . $e->getMessage()]);
    }
}

// Handle export requests
function handleExportRequest($input) {
    $report_type = isset($input['report_type']) ? $input['report_type'] : '';
    $format = isset($input['format']) ? $input['format'] : 'pdf';
    
    try {
        // Generate the report data based on report type
        $report_data = [];
        
        switch ($report_type) {
            case 'financial':
                $period = isset($input['period']) ? $input['period'] : 'monthly';
                $year = isset($input['year']) ? $input['year'] : date('Y');
                $report_data = [
                    'income' => getIncomeData($period, $year),
                    'expenses' => getExpenseData($period, $year),
                    'balance_sheet' => getBalanceSheetData(),
                    'bank_reconciliation' => getBankReconciliationData()
                ];
                break;
            
            case 'periodic':
                $report_type_param = isset($input['report_type_param']) ? $input['report_type_param'] : 'monthly-contribution';
                $period = isset($input['period']) ? $input['period'] : 'monthly';
                
                switch ($report_type_param) {
                    case 'monthly-contribution':
                        $report_data = getMonthlyContributionData($period);
                        break;
                    case 'monthly-loan':
                        $report_data = getMonthlyLoanData($period);
                        break;
                    case 'yearly-financial':
                        $report_data = getYearlyFinancialData();
                        break;
                    case 'attendance':
                        $report_data = getAttendanceData();
                        break;
                    default:
                        $report_data = [];
                }
                break;
            
            case 'analytics':
                $date_range = isset($input['date_range']) ? $input['date_range'] : '30';
                $report_data = [
                    'contribution_trends' => getContributionTrends($date_range),
                    'top_contributors' => getTopContributors(),
                    'loan_distribution' => getLoanDistribution(),
                    'income_expense_comparison' => getIncomeExpenseComparison(),
                    'member_growth' => getMemberGrowth($date_range),
                    'project_cost_breakdown' => getProjectCostBreakdown()
                ];
                break;
            
            case 'member':
                $member_id = isset($input['member_id']) ? $input['member_id'] : null;
                $date_range = isset($input['date_range']) ? $input['date_range'] : 'all';
                if ($member_id) {
                    $report_data = [
                        'member_info' => getMemberInfo($member_id),
                        'contributions' => getMemberContributions($member_id, $date_range),
                        'loans' => getMemberLoans($member_id),
                        'fines' => getMemberFines($member_id, $date_range),
                        'dividends' => getMemberDividends($member_id, $date_range)
                    ];
                }
                break;
        }
        
        // Generate export file
        $filename = generateExportFile($report_type, $format, $report_data);
        
        if ($filename) {
            http_response_code(200);
            echo json_encode([
                'message' => "{$report_type} report exported as {$format} successfully!",
                'status' => 'success',
                'filename' => $filename,
                'download_url' => "exports/{$filename}"
            ]);
        } else {
            throw new Exception('Failed to generate export file');
        }
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode([
            'error' => 'Error exporting report: ' . $e->getMessage(),
            'status' => 'error'
        ]);
    }
}

// Helper functions for data retrieval






// Get member information
function getMemberInfo($member_id) {
    try {
        $sql = "SELECT first_name, last_name, member_id, email, phone, registration_date 
                FROM members 
                WHERE id = ?";
        return fetchSingleRow($sql, [$member_id]);
    } catch (Exception $e) {
        error_log('Error in getMemberInfo: ' . $e->getMessage());
        return null;
    }
}

// Get member contributions
function getMemberContributions($member_id, $date_range = 'all') {
    try {
        // Build date condition
        $date_condition = '';
        $params = [$member_id];
        
        if ($date_range !== 'all') {
            switch ($date_range) {
                case '30':
                    $date_condition = " AND contribution_date >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)";
                    break;
                case '90':
                    $date_condition = " AND contribution_date >= DATE_SUB(CURDATE(), INTERVAL 90 DAY)";
                    break;
                case '365':
                    $date_condition = " AND contribution_date >= DATE_SUB(CURDATE(), INTERVAL 1 YEAR)";
                    break;
                default:
                    if (strpos($date_range, ',') !== false) {
                        list($from_date, $to_date) = explode(',', $date_range);
                        $date_condition = " AND contribution_date BETWEEN ? AND ?";
                        $params[] = $from_date;
                        $params[] = $to_date;
                    }
            }
        }
        
        $sql = "SELECT amount, contribution_date, description, status 
                FROM contributions 
                WHERE member_id = ? 
                $date_condition
                ORDER BY contribution_date DESC";
        
        return executeQuery($sql, $params);
    } catch (Exception $e) {
        error_log('Error in getMemberContributions: ' . $e->getMessage());
        return [];
    }
}

// Get member loans
function getMemberLoans($member_id) {
    try {
        $sql = "SELECT reference, principal_amount, interest_rate, total_amount, 
                       monthly_payment, first_payment_date, status 
                FROM loans 
                WHERE member_id = ? 
                ORDER BY application_date DESC";
        
        return executeQuery($sql, [$member_id]);
    } catch (Exception $e) {
        error_log('Error in getMemberLoans: ' . $e->getMessage());
        return [];
    }
}

// Get member fines
function getMemberFines($member_id, $date_range = 'all') {
    try {
        // Build date condition
        $date_condition = '';
        $params = [$member_id];
        
        if ($date_range !== 'all') {
            switch ($date_range) {
                case '30':
                    $date_condition = " AND fine_date >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)";
                    break;
                case '90':
                    $date_condition = " AND fine_date >= DATE_SUB(CURDATE(), INTERVAL 90 DAY)";
                    break;
                case '365':
                    $date_condition = " AND fine_date >= DATE_SUB(CURDATE(), INTERVAL 1 YEAR)";
                    break;
                default:
                    if (strpos($date_range, ',') !== false) {
                        list($from_date, $to_date) = explode(',', $date_range);
                        $date_condition = " AND fine_date BETWEEN ? AND ?";
                        $params[] = $from_date;
                        $params[] = $to_date;
                    }
            }
        }
        
        $sql = "SELECT amount, reason, description, fine_date, status 
                FROM fines 
                WHERE member_id = ? 
                $date_condition
                ORDER BY fine_date DESC";
        
        return executeQuery($sql, $params);
    } catch (Exception $e) {
        error_log('Error in getMemberFines: ' . $e->getMessage());
        return [];
    }
}

// Get member dividends
function getMemberDividends($member_id, $date_range = 'all') {
    try {
        // Build date condition
        $date_condition = '';
        $params = [$member_id];
        
        if ($date_range !== 'all') {
            switch ($date_range) {
                case '30':
                    $date_condition = " AND payment_date >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)";
                    break;
                case '90':
                    $date_condition = " AND payment_date >= DATE_SUB(CURDATE(), INTERVAL 90 DAY)";
                    break;
                case '365':
                    $date_condition = " AND payment_date >= DATE_SUB(CURDATE(), INTERVAL 1 YEAR)";
                    break;
                default:
                    if (strpos($date_range, ',') !== false) {
                        list($from_date, $to_date) = explode(',', $date_range);
                        $date_condition = " AND payment_date BETWEEN ? AND ?";
                        $params[] = $from_date;
                        $params[] = $to_date;
                    }
            }
        }
        
        $sql = "SELECT amount, financial_year, payment_date, status 
                FROM dividends 
                WHERE member_id = ? 
                $date_condition
                ORDER BY payment_date DESC";
        
        return executeQuery($sql, $params);
    } catch (Exception $e) {
        error_log('Error in getMemberDividends: ' . $e->getMessage());
        return [];
    }
}

function getContributionTrends($date_range) {
    try {
        // Calculate date range
        $date_condition = '';
        $params = [];
        
        switch ($date_range) {
            case '30':
                $date_condition = " AND contribution_date >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)";
                break;
            case '90':
                $date_condition = " AND contribution_date >= DATE_SUB(CURDATE(), INTERVAL 90 DAY)";
                break;
            case '365':
                $date_condition = " AND contribution_date >= DATE_SUB(CURDATE(), INTERVAL 1 YEAR)";
                break;
            default:
                $date_condition = " AND contribution_date >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)";
        }
        
        // Get contribution trends grouped by date
        $sql = "SELECT DATE(contribution_date) as date, SUM(amount) as amount 
                FROM contributions 
                WHERE status = 'Confirmed' 
                $date_condition
                GROUP BY DATE(contribution_date) 
                ORDER BY contribution_date";
        
        $result = executeQuery($sql, $params);
        return $result;
    } catch (Exception $e) {
        error_log('Error in getContributionTrends: ' . $e->getMessage());
        return [];
    }
}

function getTopContributors() {
    try {
        // Get top 10 contributing members
        $sql = "SELECT CONCAT(m.first_name, ' ', m.last_name) as member, 
                       SUM(c.amount) as amount
                FROM contributions c
                JOIN members m ON c.member_id = m.id
                WHERE c.status = 'Confirmed'
                GROUP BY c.member_id, m.first_name, m.last_name
                ORDER BY SUM(c.amount) DESC
                LIMIT 10";
        
        $result = executeQuery($sql);
        return $result;
    } catch (Exception $e) {
        error_log('Error in getTopContributors: ' . $e->getMessage());
        return [];
    }
}

function getLoanDistribution() {
    try {
        // Get loan distribution by purpose
        $sql = "SELECT loan_type as purpose, 
                       SUM(principal_amount) as amount
                FROM loans
                WHERE status IN ('Active', 'Completed')
                GROUP BY loan_type
                ORDER BY SUM(principal_amount) DESC";
        
        $result = executeQuery($sql);
        return $result;
    } catch (Exception $e) {
        error_log('Error in getLoanDistribution: ' . $e->getMessage());
        return [];
    }
}

function getIncomeExpenseComparison() {
    try {
        // Get total income (contributions + loan repayments + fines)
        $income_sql = "SELECT 
                        COALESCE(SUM(CASE WHEN table_name = 'contributions' THEN old_values->>'$.amount' ELSE 0 END), 0) as contributions,
                        COALESCE(SUM(CASE WHEN table_name = 'loan_repayments' THEN old_values->>'$.amount' ELSE 0 END), 0) as loan_repayments,
                        COALESCE(SUM(CASE WHEN table_name = 'fines' THEN old_values->>'$.amount' ELSE 0 END), 0) as fines
                      FROM audit_logs 
                      WHERE action = 'INSERT' 
                      AND table_name IN ('contributions', 'loan_repayments', 'fines')
                      AND created_at >= DATE_SUB(NOW(), INTERVAL 1 MONTH)";
        
        $income_result = fetchSingleRow($income_sql);
        
        // For a simpler approach, let's calculate from the actual tables
        $total_income = 0;
        $total_expenses = 0;
        
        // Get total contributions
        $contributions_sql = "SELECT COALESCE(SUM(amount), 0) as total FROM contributions WHERE status = 'Confirmed'";
        $contributions_result = fetchSingleRow($contributions_sql);
        $total_income += $contributions_result['total'];
        
        // Get total loan repayments
        $repayments_sql = "SELECT COALESCE(SUM(amount), 0) as total FROM loan_repayments WHERE status = 'Completed'";
        $repayments_result = fetchSingleRow($repayments_sql);
        $total_income += $repayments_result['total'];
        
        // Get total fines collected
        $fines_sql = "SELECT COALESCE(SUM(amount), 0) as total FROM fines WHERE status = 'Paid'";
        $fines_result = fetchSingleRow($fines_sql);
        $total_income += $fines_result['total'];
        
        // Get total expenses (simplified approach)
        // In a real implementation, you would have an expenses table
        $expenses_sql = "SELECT COALESCE(SUM(amount), 0) as total FROM payments WHERE status = 'Completed' AND account_id IN (SELECT id FROM accounts WHERE account_type_id = 6)";
        $expenses_result = fetchSingleRow($expenses_sql);
        $total_expenses = $expenses_result['total'];
        
        return [
            'income' => floatval($total_income),
            'expenses' => floatval($total_expenses)
        ];
    } catch (Exception $e) {
        error_log('Error in getIncomeExpenseComparison: ' . $e->getMessage());
        return [
            'income' => 0,
            'expenses' => 0
        ];
    }
}

// Generate export file
function generateExportFile($report_type, $format, $data) {
    // Create exports directory if it doesn't exist
    $export_dir = __DIR__ . '/../exports';
    if (!file_exists($export_dir)) {
        mkdir($export_dir, 0755, true);
    }
    
    // Generate filename
    $timestamp = date('Y-m-d_H-i-s');
    $filename = "{$report_type}_report_{$timestamp}.{$format}";
    $filepath = "{$export_dir}/{$filename}";
    
    if ($format === 'pdf') {
        // Include our PDF utility functions
        require_once __DIR__ . '/pdf-utils.php';
        
        // Generate HTML content for the PDF
        $html = generateReportHtmlForExport($report_type, $data);
        
        // Generate PDF using our utility function
        try {
            $pdf_filepath = __DIR__ . '/../exports/' . str_replace('.pdf', '', $filename) . '.pdf';
            generatePdfFromHtml($html, str_replace('.pdf', '', $filename), [], 'F');
            return $filename;
        } catch (Exception $e) {
            error_log('PDF Generation Error: ' . $e->getMessage());
            // Fallback to simple text file
            $content = "MFT GROUP - {$report_type} Report\n";
            $content .= "Generated on: " . date('Y-m-d H:i:s') . "\n\n";
            $content .= json_encode($data, JSON_PRETTY_PRINT);
            file_put_contents($filepath, $content);
        }
    } else if ($format === 'excel') {
        // For Excel, we would use a library like PhpSpreadsheet
        // For now, we'll create a CSV file as a placeholder
        $content = "MFT GROUP,{$report_type} Report\n";
        $content .= "Generated on," . date('Y-m-d H:i:s') . "\n\n";
        $content .= json_encode($data, JSON_PRETTY_PRINT);
        
        file_put_contents($filepath, $content);
    } else {
        // Default to JSON
        file_put_contents($filepath, json_encode($data, JSON_PRETTY_PRINT));
    }
    
    return $filename;
}

/**
 * Generate HTML content for export reports
 * 
 * @param string $report_type Type of report
 * @param array $data Report data
 * @return string HTML content
 */
function generateReportHtmlForExport($report_type, $data) {
    $html = '
    <!DOCTYPE html>
    <html>
    <head>
        <meta charset="UTF-8">
        <title>MFT GROUP - ' . ucfirst(str_replace('-', ' ', $report_type)) . ' Report</title>
        <style>
            body { 
                font-family: DejaVu Sans, sans-serif; 
                font-size: 11pt; 
                margin: 20mm;
            }
            .header { 
                text-align: center; 
                margin-bottom: 20px;
                border-bottom: 2px solid #14375d;
                padding-bottom: 10px;
            }
            .header h1 { 
                color: #14375d; 
                margin: 0; 
                font-size: 18pt;
            }
            .header p { 
                color: #666; 
                margin: 5px 0;
            }
            .section { 
                margin: 20px 0; 
            }
            .section-title { 
                background-color: #e8eef8; 
                color: #14375d; 
                padding: 8px 12px; 
                font-weight: bold; 
                border-left: 5px solid #14375d; 
                margin: 15px 0 10px 0;
                text-transform: uppercase;
                font-size: 12pt;
            }
            table { 
                width: 100%; 
                border-collapse: collapse; 
                margin: 10px 0; 
            }
            th, td { 
                border: 1px solid #ccc; 
                padding: 6px 8px; 
                text-align: left; 
            }
            th { 
                background-color: #f3f6fa; 
                font-weight: bold;
            }
            .footer { 
                margin-top: 30px; 
                font-size: 9pt; 
                color: #666; 
                text-align: center; 
                border-top: 1px solid #ccc;
                padding-top: 10px;
            }
            .report-info {
                background-color: #f9f9f9;
                padding: 10px;
                border: 1px solid #eee;
                margin-bottom: 20px;
            }
        </style>
    </head>
    <body>
        <div class="header">
            <h1>MFT GROUP</h1>
            <p>' . ucfirst(str_replace('-', ' ', $report_type)) . ' Report</p>
            <p>Generated on: ' . date('F j, Y g:i A') . '</p>
        </div>';
    
    // Add report information
    $html .= '<div class="report-info">';
    $html .= '<p><strong>Report Type:</strong> ' . ucfirst(str_replace('-', ' ', $report_type)) . '</p>';
    $html .= '<p><strong>Generated:</strong> ' . date('F j, Y g:i A') . '</p>';
    $html .= '</div>';
    
    // Generate content based on report type
    switch ($report_type) {
        case 'financial':
            $html .= generateFinancialReportHtml($data);
            break;
        case 'member':
            $html .= generateMemberReportHtml($data);
            break;
        case 'periodic':
            $html .= generatePeriodicReportHtml($data);
            break;
        case 'analytics':
            $html .= generateAnalyticsReportHtml($data);
            break;
        default:
            // Generic data display
            $html .= '<div class="section">';
            $html .= '<div class="section-title">Report Data</div>';
            $html .= '<pre>' . htmlspecialchars(json_encode($data, JSON_PRETTY_PRINT)) . '</pre>';
            $html .= '</div>';
    }
    
    $html .= '
        <div class="footer">
            Confidential - Generated by MFT GROUP Financial System
        </div>
    </body>
    </html>';
    
    return $html;
}

/**
 * Generate HTML for financial report
 * 
 * @param array $data Financial report data
 * @return string HTML content
 */
function generateFinancialReportHtml($data) {
    $html = '';
    
    // Income section
    if (isset($data['income'])) {
        $html .= '<div class="section">';
        $html .= '<div class="section-title">Income Summary</div>';
        $html .= '<table>
            <tr><th>Category</th><th>Amount (KES)</th></tr>';
        
        foreach ($data['income'] as $category => $amount) {
            $html .= '<tr><td>' . ucfirst(str_replace('_', ' ', $category)) . '</td><td>' . number_format($amount, 2) . '</td></tr>';
        }
        
        $html .= '</table>';
        $html .= '</div>';
    }
    
    // Expenses section
    if (isset($data['expenses'])) {
        $html .= '<div class="section">';
        $html .= '<div class="section-title">Expense Summary</div>';
        $html .= '<table>
            <tr><th>Category</th><th>Amount (KES)</th></tr>';
        
        foreach ($data['expenses'] as $category => $amount) {
            $html .= '<tr><td>' . ucfirst(str_replace('_', ' ', $category)) . '</td><td>' . number_format($amount, 2) . '</td></tr>';
        }
        
        $html .= '</table>';
        $html .= '</div>';
    }
    
    // Balance sheet section
    if (isset($data['balance_sheet'])) {
        $html .= '<div class="section">';
        $html .= '<div class="section-title">Balance Sheet</div>';
        
        // Assets
        $html .= '<h4>Assets</h4>';
        $html .= '<table>
            <tr><th>Asset</th><th>Amount (KES)</th></tr>';
        
        foreach ($data['balance_sheet']['assets'] as $asset => $amount) {
            $html .= '<tr><td>' . ucfirst(str_replace('_', ' ', $asset)) . '</td><td>' . number_format($amount, 2) . '</td></tr>';
        }
        
        $html .= '</table>';
        
        // Liabilities
        $html .= '<h4>Liabilities</h4>';
        $html .= '<table>
            <tr><th>Liability</th><th>Amount (KES)</th></tr>';
        
        foreach ($data['balance_sheet']['liabilities'] as $liability => $amount) {
            $html .= '<tr><td>' . ucfirst(str_replace('_', ' ', $liability)) . '</td><td>' . number_format($amount, 2) . '</td></tr>';
        }
        
        $html .= '</table>';
        
        // Equity
        $html .= '<h4>Equity</h4>';
        $html .= '<table>
            <tr><th>Equity</th><th>Amount (KES)</th></tr>';
        
        foreach ($data['balance_sheet']['equity'] as $equity => $amount) {
            $html .= '<tr><td>' . ucfirst(str_replace('_', ' ', $equity)) . '</td><td>' . number_format($amount, 2) . '</td></tr>';
        }
        
        $html .= '</table>';
        $html .= '</div>';
    }
    
    return $html;
}

/**
 * Generate HTML for member report
 * 
 * @param array $data Member report data
 * @return string HTML content
 */
function generateMemberReportHtml($data) {
    $html = '';
    
    // Member information
    if (isset($data['member_info'])) {
        $html .= '<div class="section">';
        $html .= '<div class="section-title">Member Information</div>';
        $html .= '<table>
            <tr><th>Field</th><th>Value</th></tr>';
        
        foreach ($data['member_info'] as $field => $value) {
            $html .= '<tr><td>' . ucfirst(str_replace('_', ' ', $field)) . '</td><td>' . htmlspecialchars($value) . '</td></tr>';
        }
        
        $html .= '</table>';
        $html .= '</div>';
    }
    
    // Contributions
    if (isset($data['contributions']) && !empty($data['contributions'])) {
        $html .= '<div class="section">';
        $html .= '<div class="section-title">Contributions</div>';
        $html .= '<table>
            <tr><th>Date</th><th>Amount (KES)</th><th>Description</th><th>Status</th></tr>';
        
        foreach ($data['contributions'] as $contribution) {
            $html .= '<tr>
                <td>' . htmlspecialchars($contribution['contribution_date']) . '</td>
                <td>' . number_format($contribution['amount'], 2) . '</td>
                <td>' . htmlspecialchars($contribution['description'] ?? 'N/A') . '</td>
                <td>' . htmlspecialchars($contribution['status']) . '</td>
            </tr>';
        }
        
        $html .= '</table>';
        $html .= '</div>';
    }
    
    // Loans
    if (isset($data['loans']) && !empty($data['loans'])) {
        $html .= '<div class="section">';
        $html .= '<div class="section-title">Loans</div>';
        $html .= '<table>
            <tr><th>Reference</th><th>Principal (KES)</th><th>Interest Rate</th><th>Status</th></tr>';
        
        foreach ($data['loans'] as $loan) {
            $html .= '<tr>
                <td>' . htmlspecialchars($loan['reference']) . '</td>
                <td>' . number_format($loan['principal_amount'], 2) . '</td>
                <td>' . number_format($loan['interest_rate'], 2) . '%</td>
                <td>' . htmlspecialchars($loan['status']) . '</td>
            </tr>';
        }
        
        $html .= '</table>';
        $html .= '</div>';
    }
    
    // Fines
    if (isset($data['fines']) && !empty($data['fines'])) {
        $html .= '<div class="section">';
        $html .= '<div class="section-title">Fines</div>';
        $html .= '<table>
            <tr><th>Date</th><th>Amount (KES)</th><th>Reason</th><th>Status</th></tr>';
        
        foreach ($data['fines'] as $fine) {
            $html .= '<tr>
                <td>' . htmlspecialchars($fine['fine_date']) . '</td>
                <td>' . number_format($fine['amount'], 2) . '</td>
                <td>' . htmlspecialchars($fine['reason']) . '</td>
                <td>' . htmlspecialchars($fine['status']) . '</td>
            </tr>';
        }
        
        $html .= '</table>';
        $html .= '</div>';
    }
    
    // Dividends
    if (isset($data['dividends']) && !empty($data['dividends'])) {
        $html .= '<div class="section">';
        $html .= '<div class="section-title">Dividends</div>';
        $html .= '<table>
            <tr><th>Date</th><th>Amount (KES)</th><th>Financial Year</th><th>Status</th></tr>';
        
        foreach ($data['dividends'] as $dividend) {
            $html .= '<tr>
                <td>' . htmlspecialchars($dividend['payment_date']) . '</td>
                <td>' . number_format($dividend['amount'], 2) . '</td>
                <td>' . htmlspecialchars($dividend['financial_year']) . '</td>
                <td>' . htmlspecialchars($dividend['status']) . '</td>
            </tr>';
        }
        
        $html .= '</table>';
        $html .= '</div>';
    }
    
    return $html;
}

/**
 * Generate HTML for periodic report
 * 
 * @param array $data Periodic report data
 * @return string HTML content
 */
function generatePeriodicReportHtml($data) {
    $html = '';
    
    $html .= '<div class="section">';
    $html .= '<div class="section-title">Periodic Report Data</div>';
    
    if (is_array($data) && !empty($data)) {
        // Check if it's a simple array of records
        if (isset($data[0]) && is_array($data[0])) {
            // Generate table with headers from first record
            $first_record = $data[0];
            $html .= '<table><thead><tr>';
            
            foreach (array_keys($first_record) as $header) {
                $html .= '<th>' . ucfirst(str_replace('_', ' ', $header)) . '</th>';
            }
            
            $html .= '</tr></thead><tbody>';
            
            foreach ($data as $record) {
                $html .= '<tr>';
                foreach ($record as $value) {
                    $html .= '<td>' . htmlspecialchars($value) . '</td>';
                }
                $html .= '</tr>';
            }
            
            $html .= '</tbody></table>';
        } else {
            // Simple key-value pairs
            $html .= '<table>';
            foreach ($data as $key => $value) {
                $html .= '<tr><td>' . ucfirst(str_replace('_', ' ', $key)) . '</td><td>' . htmlspecialchars($value) . '</td></tr>';
            }
            $html .= '</table>';
        }
    } else {
        $html .= '<p>No data available for this report.</p>';
    }
    
    $html .= '</div>';
    
    return $html;
}

/**
 * Generate HTML for analytics report
 * 
 * @param array $data Analytics report data
 * @return string HTML content
 */
function generateAnalyticsReportHtml($data) {
    $html = '';
    
    foreach ($data as $section => $section_data) {
        $html .= '<div class="section">';
        $html .= '<div class="section-title">' . ucfirst(str_replace('_', ' ', $section)) . '</div>';
        
        if (is_array($section_data) && !empty($section_data)) {
            // Check if it's a simple array of records
            if (isset($section_data[0]) && is_array($section_data[0])) {
                // Generate table with headers from first record
                $first_record = $section_data[0];
                $html .= '<table><thead><tr>';
                
                foreach (array_keys($first_record) as $header) {
                    $html .= '<th>' . ucfirst(str_replace('_', ' ', $header)) . '</th>';
                }
                
                $html .= '</tr></thead><tbody>';
                
                foreach ($section_data as $record) {
                    $html .= '<tr>';
                    foreach ($record as $value) {
                        $html .= '<td>' . htmlspecialchars($value) . '</td>';
                    }
                    $html .= '</tr>';
                }
                
                $html .= '</tbody></table>';
            } else {
                // Simple key-value pairs
                $html .= '<table>';
                foreach ($section_data as $key => $value) {
                    if (is_numeric($value)) {
                        $value = is_float($value) ? number_format($value, 2) : $value;
                    }
                    $html .= '<tr><td>' . ucfirst(str_replace('_', ' ', $key)) . '</td><td>' . htmlspecialchars($value) . '</td></tr>';
                }
                $html .= '</table>';
            }
        } else {
            $html .= '<p>No data available for this section.</p>';
        }
        
        $html .= '</div>';
    }
    
    return $html;
}

// Get member growth data
function getMemberGrowth($date_range) {
    try {
        // Calculate date range
        $date_condition = '';
        $params = [];
        
        switch ($date_range) {
            case '30':
                $date_condition = " AND registration_date >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)";
                break;
            case '90':
                $date_condition = " AND registration_date >= DATE_SUB(CURDATE(), INTERVAL 90 DAY)";
                break;
            case '365':
                $date_condition = " AND registration_date >= DATE_SUB(CURDATE(), INTERVAL 1 YEAR)";
                break;
            default:
                $date_condition = " AND registration_date >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)";
        }
        
        // Get member growth grouped by month
        $sql = "SELECT DATE_FORMAT(registration_date, '%Y-%m') as date, 
                       COUNT(*) as members
                FROM members 
                WHERE status = 'Active' 
                $date_condition
                GROUP BY DATE_FORMAT(registration_date, '%Y-%m') 
                ORDER BY registration_date";
        
        $result = executeQuery($sql, $params);
        return $result;
    } catch (Exception $e) {
        error_log('Error in getMemberGrowth: ' . $e->getMessage());
        return [];
    }
}

// Get project cost breakdown
function getProjectCostBreakdown() {
    try {
        // Get project cost breakdown
        $sql = "SELECT name as project, 
                       share_price * total_shares as cost
                FROM projects
                WHERE status IN ('Active', 'Completed')
                ORDER BY share_price * total_shares DESC";
        
        $result = executeQuery($sql);
        return $result;
    } catch (Exception $e) {
        error_log('Error in getProjectCostBreakdown: ' . $e->getMessage());
        return [];
    }
}

// Get income data
function getIncomeData($period, $year) {
    try {
        // Build date condition based on period and year
        $date_condition = '';
        $params = [];
        
        switch ($period) {
            case 'monthly':
                $date_condition = " AND YEAR(created_at) = ? AND MONTH(created_at) = ?";
                $params = [$year, date('n')]; // Current month
                break;
            case 'quarterly':
                $date_condition = " AND YEAR(created_at) = ? AND QUARTER(created_at) = ?";
                $params = [$year, ceil(date('n') / 3)]; // Current quarter
                break;
            case 'annual':
            default:
                $date_condition = " AND YEAR(created_at) = ?";
                $params = [$year];
                break;
        }
        
        // Get total contributions
        $contributions_sql = "SELECT COALESCE(SUM(amount), 0) as total 
                              FROM contributions 
                              WHERE status = 'Confirmed' 
                              $date_condition";
        $contributions_result = fetchSingleRow($contributions_sql, $params);
        
        // Get total loan repayments
        $repayments_sql = "SELECT COALESCE(SUM(amount), 0) as total 
                          FROM loan_repayments 
                          WHERE status = 'Completed' 
                          $date_condition";
        $repayments_result = fetchSingleRow($repayments_sql, $params);
        
        // Get total fines collected
        $fines_sql = "SELECT COALESCE(SUM(amount), 0) as total 
                     FROM fines 
                     WHERE status = 'Paid' 
                     $date_condition";
        $fines_result = fetchSingleRow($fines_sql, $params);
        
        // For other income, we'll use a simplified approach
        // In a real implementation, you might have a separate income table
        $other_income = 0;
        
        return [
            'contributions' => floatval($contributions_result['total']),
            'loan_repayments' => floatval($repayments_result['total']),
            'fines' => floatval($fines_result['total']),
            'other_income' => floatval($other_income)
        ];
    } catch (Exception $e) {
        error_log('Error in getIncomeData: ' . $e->getMessage());
        return [
            'contributions' => 0,
            'loan_repayments' => 0,
            'fines' => 0,
            'other_income' => 0
        ];
    }
}

// Get expense data
function getExpenseData($period, $year) {
    try {
        // Build date condition based on period and year
        $date_condition = '';
        $params = [];
        
        switch ($period) {
            case 'monthly':
                $date_condition = " AND YEAR(created_at) = ? AND MONTH(created_at) = ?";
                $params = [$year, date('n')]; // Current month
                break;
            case 'quarterly':
                $date_condition = " AND YEAR(created_at) = ? AND QUARTER(created_at) = ?";
                $params = [$year, ceil(date('n') / 3)]; // Current quarter
                break;
            case 'annual':
            default:
                $date_condition = " AND YEAR(created_at) = ?";
                $params = [$year];
                break;
        }
        
        // Get total withdrawals (assuming these are payments from savings account)
        $withdrawals_sql = "SELECT COALESCE(SUM(amount), 0) as total 
                           FROM payments 
                           WHERE status = 'Completed' 
                           AND account_id = (SELECT id FROM accounts WHERE name = 'Savings Account')
                           $date_condition";
        $withdrawals_result = fetchSingleRow($withdrawals_sql, $params);
        
        // Get total project costs
        $project_costs_sql = "SELECT COALESCE(SUM(share_price * total_shares), 0) as total 
                             FROM projects 
                             WHERE status = 'Active'
                             $date_condition";
        $project_costs_result = fetchSingleRow($project_costs_sql, $params);
        
        // Get admin fees (simplified approach)
        $admin_fees_sql = "SELECT COALESCE(SUM(amount), 0) as total 
                          FROM payments 
                          WHERE status = 'Completed' 
                          AND account_id = (SELECT id FROM accounts WHERE account_type_id = 6) // Expense account
                          $date_condition";
        $admin_fees_result = fetchSingleRow($admin_fees_sql, $params);
        
        // For other expenses, we'll use a simplified approach
        $other_expenses = 0;
        
        return [
            'withdrawals' => floatval($withdrawals_result['total']),
            'project_costs' => floatval($project_costs_result['total']),
            'admin_fees' => floatval($admin_fees_result['total']),
            'other_expenses' => floatval($other_expenses)
        ];
    } catch (Exception $e) {
        error_log('Error in getExpenseData: ' . $e->getMessage());
        return [
            'withdrawals' => 0,
            'project_costs' => 0,
            'admin_fees' => 0,
            'other_expenses' => 0
        ];
    }
}

// Get balance sheet data
function getBalanceSheetData() {
    try {
        // Get assets data
        $assets_sql = "SELECT 
                        COALESCE(SUM(CASE WHEN name = 'Savings Account' THEN balance ELSE 0 END), 0) as cash,
                        COALESCE(SUM(CASE WHEN name = 'Loan Account' THEN balance ELSE 0 END), 0) as loans_receivable,
                        COALESCE(SUM(CASE WHEN account_type_id = 4 THEN balance ELSE 0 END), 0) as projects
                      FROM accounts 
                      WHERE is_active = 1";
        $assets_result = fetchSingleRow($assets_sql);
        
        // Get liabilities data
        $liabilities_sql = "SELECT 
                           COALESCE(SUM(principal_amount), 0) as active_loans
                          FROM loans 
                          WHERE status = 'Active'";
        $liabilities_result = fetchSingleRow($liabilities_sql);
        
        // Get equity data
        $equity_sql = "SELECT 
                        COALESCE(SUM(CASE WHEN account_type_id = 1 THEN balance ELSE 0 END), 0) as member_contributions,
                        COALESCE(SUM(CASE WHEN account_type_id = 5 THEN balance ELSE 0 END), 0) as retained_earnings
                      FROM accounts 
                      WHERE is_active = 1";
        $equity_result = fetchSingleRow($equity_sql);
        
        return [
            'assets' => [
                'cash' => floatval($assets_result['cash']),
                'loans_receivable' => floatval($assets_result['loans_receivable']),
                'projects' => floatval($assets_result['projects'])
            ],
            'liabilities' => [
                'active_loans' => floatval($liabilities_result['active_loans'])
            ],
            'equity' => [
                'member_contributions' => floatval($equity_result['member_contributions']),
                'retained_earnings' => floatval($equity_result['retained_earnings'])
            ]
        ];
    } catch (Exception $e) {
        error_log('Error in getBalanceSheetData: ' . $e->getMessage());
        return [
            'assets' => [
                'cash' => 0,
                'loans_receivable' => 0,
                'projects' => 0
            ],
            'liabilities' => [
                'active_loans' => 0
            ],
            'equity' => [
                'member_contributions' => 0,
                'retained_earnings' => 0
            ]
        ];
    }
}

// Get bank reconciliation data
function getBankReconciliationData() {
    try {
        // In a real implementation, you would get this data from your banking system
        // For now, we'll return sample data based on account balances
        $bank_balance_sql = "SELECT COALESCE(SUM(balance), 0) as total FROM accounts WHERE name = 'Savings Account'";
        $bank_balance_result = fetchSingleRow($bank_balance_sql);
        
        return [
            'bank_statement_balance' => floatval($bank_balance_result['total']),
            'book_balance' => floatval($bank_balance_result['total']),
            'outstanding_cheques' => 0,
            'deposits_in_transit' => 0,
            'bank_charges' => 0,
            'interest_earned' => 0
        ];
    } catch (Exception $e) {
        error_log('Error in getBankReconciliationData: ' . $e->getMessage());
        return [
            'bank_statement_balance' => 0,
            'book_balance' => 0,
            'outstanding_cheques' => 0,
            'deposits_in_transit' => 0,
            'bank_charges' => 0,
            'interest_earned' => 0
        ];
    }
}

// Get monthly contribution data
function getMonthlyContributionData($period) {
    try {
        $sql = "SELECT DATE_FORMAT(contribution_date, '%M') as month, 
                       SUM(amount) as amount
                FROM contributions 
                WHERE status = 'Confirmed'
                AND contribution_date >= DATE_SUB(CURDATE(), INTERVAL 3 MONTH)
                GROUP BY DATE_FORMAT(contribution_date, '%Y-%m')
                ORDER BY contribution_date";
        
        $result = executeQuery($sql);
        return $result;
    } catch (Exception $e) {
        error_log('Error in getMonthlyContributionData: ' . $e->getMessage());
        return [];
    }
}

// Get monthly loan data
function getMonthlyLoanData($period) {
    try {
        $sql = "SELECT DATE_FORMAT(disbursement_date, '%M') as month, 
                       SUM(principal_amount) as amount
                FROM loans 
                WHERE status IN ('Active', 'Completed')
                AND disbursement_date >= DATE_SUB(CURDATE(), INTERVAL 3 MONTH)
                GROUP BY DATE_FORMAT(disbursement_date, '%Y-%m')
                ORDER BY disbursement_date";
        
        $result = executeQuery($sql);
        return $result;
    } catch (Exception $e) {
        error_log('Error in getMonthlyLoanData: ' . $e->getMessage());
        return [];
    }
}

// Get yearly financial data
function getYearlyFinancialData() {
    try {
        // Get total income
        $income_sql = "SELECT 
                        COALESCE(SUM(CASE WHEN table_name = 'contributions' THEN old_values->>'$.amount' ELSE 0 END), 0) as total_income
                      FROM audit_logs 
                      WHERE action = 'INSERT' 
                      AND table_name = 'contributions'
                      AND created_at >= DATE_SUB(NOW(), INTERVAL 1 YEAR)";
        
        $income_result = fetchSingleRow($income_sql);
        
        // For a simpler approach, let's calculate from the actual tables
        $total_income = 0;
        $total_expenses = 0;
        
        // Get total contributions
        $contributions_sql = "SELECT COALESCE(SUM(amount), 0) as total FROM contributions WHERE status = 'Confirmed'";
        $contributions_result = fetchSingleRow($contributions_sql);
        $total_income += $contributions_result['total'];
        
        // Get total loan repayments
        $repayments_sql = "SELECT COALESCE(SUM(amount), 0) as total FROM loan_repayments WHERE status = 'Completed'";
        $repayments_result = fetchSingleRow($repayments_sql);
        $total_income += $repayments_result['total'];
        
        // Get total fines collected
        $fines_sql = "SELECT COALESCE(SUM(amount), 0) as total FROM fines WHERE status = 'Paid'";
        $fines_result = fetchSingleRow($fines_sql);
        $total_income += $fines_result['total'];
        
        // Get total expenses (simplified approach)
        $expenses_sql = "SELECT COALESCE(SUM(amount), 0) as total FROM payments WHERE status = 'Completed' AND account_id IN (SELECT id FROM accounts WHERE account_type_id = 6)";
        $expenses_result = fetchSingleRow($expenses_sql);
        $total_expenses = $expenses_result['total'];
        
        $net_profit = $total_income - $total_expenses;
        
        return [
            'total_income' => floatval($total_income),
            'total_expenses' => floatval($total_expenses),
            'net_profit' => floatval($net_profit)
        ];
    } catch (Exception $e) {
        error_log('Error in getYearlyFinancialData: ' . $e->getMessage());
        return [
            'total_income' => 0,
            'total_expenses' => 0,
            'net_profit' => 0
        ];
    }
}

// Get attendance data
function getAttendanceData() {
    try {
        $sql = "SELECT m.meeting_date, 
                       COUNT(CASE WHEN ma.attendance_status = 'Present' THEN 1 END) as attendees,
                       COUNT(CASE WHEN ma.attendance_status = 'Absent' THEN 1 END) as absentees
                FROM meetings m
                LEFT JOIN meeting_attendance ma ON m.id = ma.meeting_id
                WHERE m.meeting_date >= DATE_SUB(CURDATE(), INTERVAL 6 MONTH)
                GROUP BY m.id, m.meeting_date
                ORDER BY m.meeting_date DESC
                LIMIT 10";
        
        $result = executeQuery($sql);
        return $result;
    } catch (Exception $e) {
        error_log('Error in getAttendanceData: ' . $e->getMessage());
        return [];
    }
}

?>