<?php
// MFT GROUP Projects API
// This API provides project data and handles project assignments for members

// Handle preflight requests (OPTIONS)
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    // Specify the allowed methods
    header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS");
    // Specify the allowed headers
    header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");
    // Specify the allowed origin
    header("Access-Control-Allow-Origin: *");
    // Allow credentials if needed
    header("Access-Control-Allow-Credentials: true");
    // Set max age for preflight request caching
    header("Access-Control-Max-Age: 86400");
    // Exit early for OPTIONS requests
    exit(0);
}

// Enable CORS for frontend access
header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");

// Add cache control headers to prevent caching issues
header("Cache-Control: no-cache, no-store, must-revalidate");
header("Pragma: no-cache");
header("Expires: 0");

// Include database configuration
require_once __DIR__ . '/../database/config.php';

// Get request method
$method = $_SERVER['REQUEST_METHOD'];

try {
    switch ($method) {
        case 'GET':
            handleGetRequest();
            break;
        case 'POST':
            handlePostRequest();
            break;
        case 'PUT':
            handlePutRequest();
            break;
        case 'DELETE':
            handleDeleteRequest();
            break;
        case 'OPTIONS':
            // Already handled above, but just in case
            exit(0);
            break;
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            break;
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Internal server error: ' . $e->getMessage()]);
}

function handleGetRequest() {
    if (isset($_GET['id']) && isset($_GET['member_id'])) {
        // Get specific project for a member
        getProjectForMember($_GET['id'], $_GET['member_id']);
    } else if (isset($_GET['member_id'])) {
        // Get projects assigned to a specific member
        getMemberProjects($_GET['member_id']);
    } else if (isset($_GET['id'])) {
        // Get specific project
        getProject($_GET['id']);
    } else {
        // Get all projects
        getProjects();
    }
}

function handlePostRequest() {
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid JSON data']);
        return;
    }
    
    if (isset($input['action']) && $input['action'] === 'assign') {
        // Assign projects to member
        assignProjectsToMember($input);
    } else {
        // Create new project
        createProject($input);
    }
}

function handlePutRequest() {
    // Log the request for debugging
    error_log("PUT request received. Project ID: " . ($_GET['id'] ?? 'not provided'));
    
    if (!isset($_GET['id'])) {
        error_log("PUT request failed: Project ID is required");
        http_response_code(400);
        echo json_encode(['error' => 'Project ID is required']);
        return;
    }
    
    // Log raw input data
    $raw_input = file_get_contents('php://input');
    error_log("Raw input data: " . $raw_input);
    
    // Handle empty input
    if (empty($raw_input)) {
        error_log("PUT request failed: No input data provided");
        http_response_code(400);
        echo json_encode(['error' => 'No input data provided']);
        return;
    }
    
    $input = json_decode($raw_input, true);
    
    // Check for JSON parsing errors
    if (json_last_error() !== JSON_ERROR_NONE) {
        error_log("PUT request failed: Invalid JSON data. JSON error: " . json_last_error_msg());
        http_response_code(400);
        echo json_encode(['error' => 'Invalid JSON data: ' . json_last_error_msg()]);
        return;
    }
    
    // Handle case where json_decode returns null (e.g., for empty array)
    if ($input === null && $raw_input !== 'null') {
        // This shouldn't happen with valid JSON, but let's be safe
        error_log("PUT request failed: Failed to parse JSON data");
        http_response_code(400);
        echo json_encode(['error' => 'Failed to parse JSON data']);
        return;
    }
    
    error_log("Parsed input data: " . json_encode($input));
    
    updateProject($_GET['id'], $input);
}

function handleDeleteRequest() {
    if (!isset($_GET['id'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Project ID is required']);
        return;
    }
    
    deleteProject($_GET['id']);
}

function getProjects() {
    // Get filter parameters
    $status = isset($_GET['status']) ? $_GET['status'] : null;
    $search = isset($_GET['search']) ? $_GET['search'] : null;
    $date_from = isset($_GET['date_from']) ? $_GET['date_from'] : null;
    $date_to = isset($_GET['date_to']) ? $_GET['date_to'] : null;
    
    // Build WHERE clause
    $where_conditions = [];
    $params = [];
    
    if ($status && $status !== 'all') {
        $where_conditions[] = "status = ?";
        $params[] = $status;
    }
    
    if ($search && $search !== 'all') {
        $where_conditions[] = "(name LIKE ? OR description LIKE ?)";
        $search_param = '%' . $search . '%';
        $params[] = $search_param;
        $params[] = $search_param;
    }
    
    // Add date range filters (using completion_date instead of start_date)
    if ($date_from) {
        $where_conditions[] = "completion_date >= ?";
        $params[] = $date_from;
    }
    
    if ($date_to) {
        $where_conditions[] = "completion_date <= ?";
        $params[] = $date_to;
    }
    
    $where_clause = '';
    if (!empty($where_conditions)) {
        $where_clause = 'WHERE ' . implode(' AND ', $where_conditions);
    }
    
    $sql = "SELECT id, name, description, location, completion_date, status, total_shares, share_price, current_amount,
                   created_at, updated_at
            FROM projects 
            $where_clause
            ORDER BY completion_date DESC";
    
    $projects = executeQuery($sql, $params);
    
    // Transform projects to match frontend expectations
    $transformed_projects = array_map(function($project) {
        // Calculate target_amount based on total_shares and share_price
        $total_shares = (int)($project['total_shares'] ?? 0);
        $share_price = (float)($project['share_price'] ?? 0);
        $target_amount = $total_shares * $share_price;
        // Use the actual current_amount from the database instead of simulating
        $current_amount = (float)($project['current_amount'] ?? 0);
        
        return [
            'id' => $project['id'],
            'name' => $project['name'],
            'description' => $project['description'],
            'location' => $project['location'],
            'completion_date' => $project['completion_date'],
            'status' => $project['status'],
            'total_shares' => $project['total_shares'],
            'share_price' => $project['share_price'],
            'target_amount' => $target_amount,
            'current_amount' => $current_amount,
            'start_date' => date('Y-m-d', strtotime('-30 days', strtotime($project['completion_date'] ?? date('Y-m-d')))),
            'end_date' => $project['completion_date'],
            'created_at' => $project['created_at'],
            'updated_at' => $project['updated_at']
        ];
    }, $projects);
    
    http_response_code(200);
    echo json_encode([
        'count' => count($transformed_projects),
        'projects' => $transformed_projects
    ]);
}

function getProject($id) {
    // Check if member_id is provided in the request
    $member_id = isset($_GET['member_id']) ? $_GET['member_id'] : null;
    
    if ($member_id) {
        // Get project with member-specific information
        getProjectForMember($id, $member_id);
        return;
    }
    
    $sql = "SELECT id, name, description, location, completion_date, status, total_shares, share_price, current_amount,
                   created_at, updated_at
            FROM projects 
            WHERE id = ?";
    
    $project = fetchSingleRow($sql, [$id]);
    
    if ($project) {
        // Transform project to match frontend expectations
        $total_shares = (int)($project['total_shares'] ?? 0);
        $share_price = (float)($project['share_price'] ?? 0);
        $target_amount = $total_shares * $share_price;
        // Use the actual current_amount from the database instead of simulating
        $current_amount = (float)($project['current_amount'] ?? 0);
        
        $transformed_project = [
            'id' => $project['id'],
            'name' => $project['name'],
            'description' => $project['description'],
            'location' => $project['location'],
            'completion_date' => $project['completion_date'],
            'status' => $project['status'],
            'total_shares' => $project['total_shares'],
            'share_price' => $project['share_price'],
            'target_amount' => $target_amount,
            'current_amount' => $current_amount,
            'start_date' => date('Y-m-d', strtotime('-30 days', strtotime($project['completion_date'] ?? date('Y-m-d')))),
            'end_date' => $project['completion_date'],
            'created_at' => $project['created_at'],
            'updated_at' => $project['updated_at']
        ];
        
        http_response_code(200);
        echo json_encode($transformed_project);
    } else {
        http_response_code(404);
        echo json_encode(['error' => 'Project not found']);
    }
}

function getProjectForMember($project_id, $member_id) {
    // Validate member ID
    if (!$member_id) {
        http_response_code(400);
        echo json_encode(['error' => 'Member ID is required']);
        return;
    }
    
    // Check if the member_id is numeric (database ID) or a reference ID
    $actual_member_id = $member_id;
    if (!is_numeric($member_id)) {
        // It's a reference ID, look up the actual database ID
        $member_lookup_sql = "SELECT id FROM members WHERE member_id = ?";
        $member_result = fetchSingleRow($member_lookup_sql, [$member_id]);
        
        if (!$member_result) {
            http_response_code(404);
            echo json_encode(['error' => 'Member not found']);
            return;
        }
        
        // Use the actual database ID
        $actual_member_id = $member_result['id'];
    }
    
    // Get project with member-specific information
    // Fixed field names to match the database schema
    $sql = "SELECT p.id, p.name, p.description, p.location, p.completion_date, p.status, p.total_shares, p.share_price, p.current_amount,
                   mp.amount_contributed as contribution_amount, mp.join_date as joined_date, mp.status as member_project_status,
                   p.created_at, p.updated_at
            FROM projects p
            LEFT JOIN member_projects mp ON p.id = mp.project_id AND mp.member_id = ?
            WHERE p.id = ?";
    
    $project = fetchSingleRow($sql, [$actual_member_id, $project_id]);
    
    if ($project) {
        // Add a default role for frontend compatibility
        $project['role'] = 'Member'; // Default role
        
        // Calculate target_amount based on total_shares and share_price
        $project['target_amount'] = (float)$project['total_shares'] * (float)$project['share_price'];
        
        // Use the actual current_amount from the database
        $project['current_amount'] = (float)$project['current_amount'];
        
        // Add start_date for frontend compatibility
        $project['start_date'] = date('Y-m-d', strtotime('-30 days', strtotime($project['completion_date'] ?? date('Y-m-d'))));
        
        // Add end_date for frontend compatibility
        $project['end_date'] = $project['completion_date'];
        
        http_response_code(200);
        echo json_encode($project);
    } else {
        http_response_code(404);
        echo json_encode(['error' => 'Project not found']);
    }
}

function getMemberProjects($member_id) {
    // Validate member ID
    if (!$member_id) {
        http_response_code(400);
        echo json_encode(['error' => 'Member ID is required']);
        return;
    }
    
    // Check if the member_id is numeric (database ID) or a reference ID
    if (!is_numeric($member_id)) {
        // It's a reference ID, look up the actual database ID
        $member_lookup_sql = "SELECT id FROM members WHERE member_id = ?";
        $member_result = fetchSingleRow($member_lookup_sql, [$member_id]);
        
        if (!$member_result) {
            http_response_code(404);
            echo json_encode(['error' => 'Member not found']);
            return;
        }
        
        // Use the actual database ID
        $member_id = $member_result['id'];
    }
    
    // Fixed field names to match the database schema
    $sql = "SELECT p.id, p.name, p.description, p.location, p.completion_date, p.status, p.total_shares, p.share_price, p.current_amount,
                   mp.amount_contributed as contribution_amount, mp.join_date as joined_date, mp.status as member_project_status,
                   p.created_at, p.updated_at
            FROM projects p
            JOIN member_projects mp ON p.id = mp.project_id
            WHERE mp.member_id = ?
            ORDER BY p.completion_date DESC";
    
    try {
        $projects = executeQuery($sql, [$member_id]);
        
        // Transform the data to match what the frontend expects
        $transformed_projects = array_map(function($project) {
            // Add compatibility fields for frontend
            $project['role'] = 'Member'; // Default role for compatibility
            $project['target_amount'] = (float)$project['total_shares'] * (float)$project['share_price'];
            $project['current_amount'] = (float)$project['current_amount'];
            $project['start_date'] = date('Y-m-d', strtotime('-30 days', strtotime($project['completion_date'] ?? date('Y-m-d'))));
            $project['end_date'] = $project['completion_date'];
            
            return $project;
        }, $projects);
        
        http_response_code(200);
        echo json_encode([
            'count' => count($transformed_projects),
            'projects' => $transformed_projects
        ]);
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Database error: ' . $e->getMessage()]);
    }
}

function createProject($input) {
    // Validate required fields
    $required_fields = ['name'];
    foreach ($required_fields as $field) {
        if (!isset($input[$field]) || empty($input[$field])) {
            http_response_code(400);
            echo json_encode(['error' => "Missing required field: $field"]);
            return;
        }
    }
    
    // Map frontend fields to database fields
    $name = $input['name'];
    $description = $input['description'] ?? null;
    $location = $input['location'] ?? null;
    $completion_date = $input['completion_date'] ?? null;
    $status = $input['status'] ?? 'Planning';
    $current_amount = $input['current_amount'] ?? 0.00;  // Add current_amount
    
    // Handle target_amount and share_price calculation
    $total_shares = 0;
    $share_price = 0.00;
    
    if (isset($input['target_amount']) && is_numeric($input['target_amount'])) {
        $target_amount = (float)$input['target_amount'];
        if (isset($input['share_price']) && is_numeric($input['share_price']) && $input['share_price'] > 0) {
            // Both target_amount and share_price provided
            $share_price = (float)$input['share_price'];
            $total_shares = (int)($target_amount / $share_price);
        } else if (isset($input['total_shares']) && is_numeric($input['total_shares']) && $input['total_shares'] > 0) {
            // target_amount and total_shares provided
            $total_shares = (int)$input['total_shares'];
            $share_price = $target_amount / $total_shares;
        } else {
            // Only target_amount provided, use default values
            $share_price = 1000.00; // Default share price
            $total_shares = (int)($target_amount / $share_price);
        }
    } else {
        // Use provided values or defaults
        $total_shares = $input['total_shares'] ?? 0;
        $share_price = $input['share_price'] ?? 0.00;
    }
    
    $sql = "INSERT INTO projects (name, description, location, completion_date, total_shares, share_price, current_amount, status) 
            VALUES (?, ?, ?, ?, ?, ?, ?, ?)";
    
    $params = [
        $name,
        $description,
        $location,
        $completion_date,
        $total_shares,
        $share_price,
        $current_amount,
        $status
    ];
    
    try {
        beginTransaction();
        
        $project_id = executeInsert($sql, $params);
        
        commitTransaction();
        
        // Fetch the created project data
        $project_sql = "SELECT id, name, description, location, completion_date, status, total_shares, share_price, current_amount, created_at, updated_at FROM projects WHERE id = ?";
        $created_project = fetchSingleRow($project_sql, [$project_id]);
        
        // Transform project to match frontend expectations
        if ($created_project) {
            $total_shares = (int)($created_project['total_shares'] ?? 0);
            $share_price = (float)($created_project['share_price'] ?? 0);
            $target_amount = $total_shares * $share_price;
            // Use the actual current_amount from the database instead of simulating
            $current_amount = (float)($created_project['current_amount'] ?? 0);
            
            $transformed_project = [
                'id' => $created_project['id'],
                'name' => $created_project['name'],
                'description' => $created_project['description'],
                'location' => $created_project['location'],
                'completion_date' => $created_project['completion_date'],
                'status' => $created_project['status'],
                'total_shares' => $created_project['total_shares'],
                'share_price' => $created_project['share_price'],
                'target_amount' => $target_amount,
                'current_amount' => $current_amount,
                'start_date' => date('Y-m-d', strtotime('-30 days', strtotime($created_project['completion_date'] ?? date('Y-m-d')))),
                'end_date' => $created_project['completion_date'],
                'created_at' => $created_project['created_at'],
                'updated_at' => $created_project['updated_at']
            ];
        } else {
            $transformed_project = null;
        }
        
        http_response_code(201);
        echo json_encode([
            'message' => 'Project created successfully',
            'project_id' => $project_id,
            'project' => $transformed_project
        ]);
    } catch (Exception $e) {
        rollbackTransaction();
        error_log("Error creating project: " . $e->getMessage());
        http_response_code(500);
        echo json_encode(['error' => 'Failed to create project: ' . $e->getMessage()]);
        return;
    }
}

function updateProject($id, $input) {
    // Log the input for debugging
    error_log("Update project $id with input: " . json_encode($input));
    
    // Build dynamic update query based on provided fields
    $updates = [];
    $params = [];
    
    // Map frontend fields to database fields
    $field_mapping = [
        'name' => 'name',
        'description' => 'description',
        'location' => 'location',
        'completion_date' => 'completion_date',
        'status' => 'status',
        'total_shares' => 'total_shares',
        'share_price' => 'share_price',
        'current_amount' => 'current_amount'  // Add current_amount to field mapping
    ];
    
    // Handle special case: if target_amount is provided, we need to calculate total_shares or share_price
    if (isset($input['target_amount']) && is_numeric($input['target_amount'])) {
        // If we have target_amount, we need to determine how to split it between total_shares and share_price
        // For now, we'll assume share_price is fixed and calculate total_shares
        $target_amount = (float)$input['target_amount'];
        if (isset($input['share_price']) && is_numeric($input['share_price']) && $input['share_price'] > 0) {
            $share_price = (float)$input['share_price'];
            $total_shares = (int)($target_amount / $share_price);
            $updates[] = "total_shares = ?";
            $params[] = $total_shares;
            error_log("Calculated total_shares: $total_shares from target_amount: $target_amount and share_price: $share_price");
        } else {
            // If no share_price provided, we'll use the existing one from database or default
            $existing_project = fetchSingleRow("SELECT share_price FROM projects WHERE id = ?", [$id]);
            if ($existing_project && $existing_project['share_price'] > 0) {
                $share_price = (float)$existing_project['share_price'];
                $total_shares = (int)($target_amount / $share_price);
                $updates[] = "total_shares = ?";
                $params[] = $total_shares;
                error_log("Calculated total_shares: $total_shares from target_amount: $target_amount and existing share_price: $share_price");
            }
        }
    }
    
    // Add standard fields that map directly to database columns
    foreach ($field_mapping as $frontend_field => $db_field) {
        // Check if the field exists and is not null or empty string
        if (isset($input[$frontend_field]) && $input[$frontend_field] !== '' && $input[$frontend_field] !== null) {
            $updates[] = "$db_field = ?";
            $params[] = $input[$frontend_field];
            error_log("Adding field '$db_field' with value: " . json_encode($input[$frontend_field]));
        }
    }
    
    // Handle start_date mapping (frontend sends start_date, but we store it differently)
    // For now, we'll ignore start_date as it's calculated from completion_date in the response
    
    // Only proceed if there are fields to update
    if (empty($updates)) {
        error_log("No valid fields to update for project $id");
        http_response_code(400);
        echo json_encode(['error' => 'No valid fields to update']);
        return;
    }
    
    // Add the ID for the WHERE clause
    $params[] = $id;
    
    $sql = "UPDATE projects SET " . implode(', ', $updates) . ", updated_at = CURRENT_TIMESTAMP WHERE id = ?";
    
    error_log("Executing update query: $sql with params: " . json_encode($params));
    
    try {
        beginTransaction();
        
        $result = executeNonQuery($sql, $params);
        error_log("Update query affected $result rows");
        
        // Check if any rows were actually updated
        if ($result === 0) {
            error_log("Warning: No rows were updated for project $id. This might mean the data was the same or the project doesn't exist.");
        }
        
        commitTransaction();
        
        // Also fetch the updated project data to return
        $project_sql = "SELECT id, name, description, location, completion_date, status, total_shares, share_price, current_amount, created_at, updated_at FROM projects WHERE id = ?";
        $updated_project = fetchSingleRow($project_sql, [$id]);
        
        // Transform project to match frontend expectations
        if ($updated_project) {
            $total_shares = (int)($updated_project['total_shares'] ?? 0);
            $share_price = (float)($updated_project['share_price'] ?? 0);
            $target_amount = $total_shares * $share_price;
            // Use the actual current_amount from the database instead of simulating
            $current_amount = (float)($updated_project['current_amount'] ?? 0);
            
            $transformed_project = [
                'id' => $updated_project['id'],
                'name' => $updated_project['name'],
                'description' => $updated_project['description'],
                'location' => $updated_project['location'],
                'completion_date' => $updated_project['completion_date'],
                'status' => $updated_project['status'],
                'total_shares' => $updated_project['total_shares'],
                'share_price' => $updated_project['share_price'],
                'target_amount' => $target_amount,
                'current_amount' => $current_amount,
                'start_date' => date('Y-m-d', strtotime('-30 days', strtotime($updated_project['completion_date'] ?? date('Y-m-d')))),
                'end_date' => $updated_project['completion_date'], // Map completion_date to end_date for frontend
                'created_at' => $updated_project['created_at'],
                'updated_at' => $updated_project['updated_at']
            ];
        } else {
            $transformed_project = null;
        }
        
        http_response_code(200);
        echo json_encode([
            'message' => 'Project updated successfully',
            'rows_affected' => $result,
            'project' => $transformed_project
        ]);
    } catch (Exception $e) {
        error_log("Error updating project $id: " . $e->getMessage());
        rollbackTransaction();
        http_response_code(500);
        echo json_encode(['error' => 'Failed to update project: ' . $e->getMessage()]);
        return;
    }
}

function deleteProject($id) {
    try {
        beginTransaction();
        
        // Delete member_project associations first (due to foreign key constraints)
        $sql1 = "DELETE FROM member_projects WHERE project_id = ?";
        executeNonQuery($sql1, [$id]);
        
        // Delete the project
        $sql2 = "DELETE FROM projects WHERE id = ?";
        executeNonQuery($sql2, [$id]);
        
        commitTransaction();
        
        http_response_code(200);
        echo json_encode([
            'message' => 'Project deleted successfully'
        ]);
    } catch (Exception $e) {
        rollbackTransaction();
        throw $e;
    }
}

function assignProjectsToMember($input) {
    if (!isset($input['member_id']) || !isset($input['project_ids'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Member ID and project IDs are required']);
        return;
    }
    
    $member_id = $input['member_id'];
    $project_ids = $input['project_ids'];
    $joined_date = $input['joined_date'] ?? date('Y-m-d');
    $contribution_amount = $input['contribution_amount'] ?? 0.00;
    $role = $input['role'] ?? 'Member'; // Get role from input, default to 'Member'
    
    try {
        beginTransaction();
        
        // Assign new projects
        foreach ($project_ids as $project_id) {
            // Check if assignment already exists
            $check_sql = "SELECT id FROM member_projects WHERE member_id = ? AND project_id = ?";
            $existing = fetchSingleRow($check_sql, [$member_id, $project_id]);
            
            if ($existing) {
                // Update existing assignment (fixed: removed non-existent columns)
                $update_sql = "UPDATE member_projects SET 
                              amount_contributed = ?, 
                              join_date = ?, 
                              status = 'Active',
                              updated_at = CURRENT_TIMESTAMP
                              WHERE member_id = ? AND project_id = ?";
                executeNonQuery($update_sql, [$contribution_amount, $joined_date, $member_id, $project_id]);
            } else {
                // Create new assignment (fixed: removed non-existent columns)
                $insert_sql = "INSERT INTO member_projects (member_id, project_id, amount_contributed, join_date, status) 
                               VALUES (?, ?, ?, ?, ?)";
                executeNonQuery($insert_sql, [$member_id, $project_id, $contribution_amount, $joined_date, 'Active']);
            }
            
            // Update the project's current_amount by summing all contributions
            $sum_sql = "SELECT COALESCE(SUM(amount_contributed), 0) as total_contributions FROM member_projects WHERE project_id = ?";
            $sum_result = fetchSingleRow($sum_sql, [$project_id]);
            $total_contributions = $sum_result['total_contributions'];
            
            // Update the project with the new total contributions
            $update_project_sql = "UPDATE projects SET current_amount = ? WHERE id = ?";
            executeNonQuery($update_project_sql, [$total_contributions, $project_id]);
        }
        
        commitTransaction();
        
        http_response_code(200);
        echo json_encode([
            'message' => 'Projects assigned to member successfully',
            'member_id' => $member_id,
            'assigned_projects' => count($project_ids),
            'role' => $role // Return the role for frontend
        ]);
        
        // Trigger real-time update to member dashboard
        triggerMemberDashboardUpdate($member_id, $project_ids);
        
    } catch (Exception $e) {
        rollbackTransaction();
        throw $e;
    }
}

function triggerMemberDashboardUpdate($member_id, $project_ids) {
    // In a real implementation, this would send a WebSocket message or similar
    // to update the member's dashboard in real-time
    error_log("Triggering real-time update for member $member_id with projects: " . json_encode($project_ids));
}
?>