<?php
// MFT GROUP Officials API
// This API provides current officials data for the frontend

// Enable CORS for frontend access
header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Methods: GET");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");

// Include database configuration
require_once __DIR__ . '/../database/config.php';

// Get request method
$method = $_SERVER['REQUEST_METHOD'];

if ($method !== 'GET') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit;
}

try {
    // Get current officials with their roles
    $officials = getCurrentOfficials();
    
    // Always return a 200 status with the officials data (even if empty)
    http_response_code(200);
    echo json_encode([
        'status' => 'success',
        'officials' => $officials
    ]);
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'status' => 'error',
        'error' => 'Internal server error: ' . $e->getMessage()
    ]);
}

function getCurrentOfficials() {
    try {
        $pdo = getDatabaseConnection();
        
        // Define the roles we want to display and their display names
        $roleMapping = [
            'Chairman' => 'Chairman',
            'Assistant Chairman' => 'Assistant Chairman',
            'Secretary' => 'Secretary',
            'Assistant Secretary' => 'Organising Secretary',
            'Treasurer' => 'Treasurer',
            'Loans Officer' => 'Loans Officer'
        ];
        
        // Get members with their roles
        $sql = "SELECT 
                    m.id,
                    m.member_id,
                    m.first_name,
                    m.last_name,
                    m.email,
                    m.phone,
                    m.profile_picture,
                    r.name as role_name
                FROM members m
                JOIN member_roles mr ON m.id = mr.member_id
                JOIN roles r ON mr.role_id = r.id
                WHERE m.status = 'Active'
                AND r.name IN ('Chairman', 'Assistant Chairman', 'Secretary', 'Assistant Secretary', 'Treasurer', 'Loans Officer')
                ORDER BY FIELD(r.name, 'Chairman', 'Assistant Chairman', 'Secretary', 'Assistant Secretary', 'Treasurer', 'Loans Officer')";
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute();
        $results = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Format the results
        $officials = [];
        foreach ($results as $result) {
            $officials[] = [
                'id' => $result['id'],
                'member_id' => $result['member_id'],
                'name' => $result['first_name'] . ' ' . $result['last_name'],
                'first_name' => $result['first_name'],
                'last_name' => $result['last_name'],
                'email' => $result['email'],
                'phone' => $result['phone'],
                'profile_photo' => $result['profile_picture'],
                'role' => $result['role_name'],
                'display_role' => $roleMapping[$result['role_name']] ?? $result['role_name']
            ];
        }
        
        // Always return an array, even if empty
        return $officials ?: [];
    } catch (Exception $e) {
        error_log("Error in getCurrentOfficials: " . $e->getMessage());
        throw new Exception("Database error: " . $e->getMessage());
    }
}
?>