<?php
// MFT GROUP Notification Service Class
// This script contains only the notification service class definition

require_once __DIR__ . '/../database/config.php';

class MFTNotificationService {
    private $pdo;
    
    public function __construct() {
        $this->pdo = getDatabaseConnection();
    }
    
    /**
     * Send a notification to all connected clients
     * @param string $type The type of notification (e.g., 'settings_update', 'contribution_added')
     * @param array $data The data to send with the notification
     * @param string $target The target audience ('admin', 'member_dashboard', 'all')
     * @param int $memberId Optional member ID for member-specific notifications
     * @return bool Success status
     */
    public function sendNotification($type, $data = [], $target = 'all', $memberId = null) {
        try {
            // Insert notification into database for polling clients
            $sql = "INSERT INTO notifications (type, data, target, member_id, created_at) VALUES (?, ?, ?, ?, NOW())";
            $stmt = $this->pdo->prepare($sql);
            $jsonData = json_encode($data);
            $stmt->execute([$type, $jsonData, $target, $memberId]);
            
            // Also send to WebSocket server if available
            $this->sendToWebSocketServer($type, $data, $target, $memberId);
            
            error_log("Notification sent: $type to $target");
            return true;
        } catch (Exception $e) {
            error_log("Error sending notification: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Send notification to WebSocket server
     * @param string $type
     * @param array $data
     * @param string $target
     * @param int $memberId
     */
    private function sendToWebSocketServer($type, $data, $target, $memberId) {
        // In a real implementation, you would connect to your WebSocket server
        // This is a placeholder for now
        error_log("Sending to WebSocket: $type");
    }
    
    /**
     * Get pending notifications for a client
     * @param string $target
     * @param int $memberId
     * @param int $lastCheck Timestamp of last check
     * @return array Notifications
     */
    public function getNotifications($target = 'all', $memberId = null, $lastCheck = 0) {
        try {
            $sql = "SELECT id, type, data, created_at FROM notifications 
                    WHERE (target = ? OR target = 'all') 
                    AND created_at > FROM_UNIXTIME(?)";
            $params = [$target, $lastCheck];
            
            // Fix the target check for member dashboard
            if ($memberId && $target === 'member_dashboard') {
                $sql .= " AND (member_id = ? OR member_id IS NULL)";
                $params[] = $memberId;
            }
            
            $sql .= " ORDER BY created_at ASC LIMIT 50";
            
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute($params);
            $notifications = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            // Mark notifications as delivered
            if (!empty($notifications)) {
                $ids = array_column($notifications, 'id');
                $placeholders = str_repeat('?,', count($ids) - 1) . '?';
                $updateSql = "UPDATE notifications SET delivered_at = NOW() WHERE id IN ($placeholders)";
                $updateStmt = $this->pdo->prepare($updateSql);
                $updateStmt->execute($ids);
            }
            
            // Decode JSON data
            foreach ($notifications as &$notification) {
                $notification['data'] = json_decode($notification['data'], true);
            }
            
            return $notifications;
        } catch (Exception $e) {
            error_log("Error getting notifications: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Clean old notifications
     * @param int $days Number of days to keep notifications
     */
    public function cleanupOldNotifications($days = 7) {
        try {
            $sql = "DELETE FROM notifications WHERE created_at < DATE_SUB(NOW(), INTERVAL ? DAY)";
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute([$days]);
            return $stmt->rowCount();
        } catch (Exception $e) {
            error_log("Error cleaning notifications: " . $e->getMessage());
            return 0;
        }
    }
}
?>