<?php
// MFT GROUP Missed Contributions API
// This API handles missed contribution records and tracking

// Enable CORS for frontend access
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Credentials: true");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS");
header("Access-Control-Allow-Headers: Origin, X-Requested-With, Content-Type, Accept, Authorization");
header("Content-Type: application/json; charset=UTF-8");

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Include database configuration
require_once dirname(__DIR__) . '/database/config.php';

// Get request method
$method = $_SERVER['REQUEST_METHOD'];

try {
    switch ($method) {
        case 'GET':
            handleGetRequest();
            break;
        case 'POST':
            handlePostRequest();
            break;
        case 'PUT':
            handlePutRequest();
            break;
        case 'DELETE':
            handleDeleteRequest();
            break;
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            break;
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Internal server error: ' . $e->getMessage()]);
}

function handleGetRequest() {
    // Get query parameters
    $member_id = isset($_GET['member_id']) ? $_GET['member_id'] : null;
    $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 10;
    $offset = isset($_GET['offset']) ? (int)$_GET['offset'] : 0;
    
    if ($member_id) {
        // Get missed contributions for a specific member
        getMemberMissedContributions($member_id, $limit, $offset);
    } else {
        // Get all missed contributions (admin only)
        getAllMissedContributions($limit, $offset);
    }
}

function handlePostRequest() {
    // Record a new missed contribution
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid JSON data']);
        return;
    }
    
    recordMissedContribution($input);
}

function handlePutRequest() {
    // Update a missed contribution record (e.g., when payment is made)
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input || !isset($input['id'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Missing required fields']);
        return;
    }
    
    updateMissedContribution($input);
}

function handleDeleteRequest() {
    // Delete a missed contribution record
    $id = isset($_GET['id']) ? (int)$_GET['id'] : null;
    
    if (!$id) {
        http_response_code(400);
        echo json_encode(['error' => 'Missing ID parameter']);
        return;
    }
    
    deleteMissedContribution($id);
}

function getMemberMissedContributions($member_id, $limit, $offset) {
    global $pdo;
    
    try {
        // First, get the member's numeric ID
        $member_check_sql = "SELECT id FROM members WHERE member_id = ?";
        $member_stmt = $pdo->prepare($member_check_sql);
        $member_stmt->execute([$member_id]);
        $member_result = $member_stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$member_result) {
            http_response_code(404);
            echo json_encode(['error' => 'Member not found']);
            return;
        }
        
        $member_numeric_id = $member_result['id'];
        
        // Get missed contributions for this member
        $sql = "SELECT mc.*, m.first_name, m.last_name, m.member_id 
                FROM missed_contributions mc
                JOIN members m ON mc.member_id = m.id
                WHERE mc.member_id = ?
                ORDER BY mc.missed_date DESC
                LIMIT ? OFFSET ?";
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute([$member_numeric_id, $limit, $offset]);
        $results = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Get total count for pagination
        $count_sql = "SELECT COUNT(*) as total FROM missed_contributions WHERE member_id = ?";
        $count_stmt = $pdo->prepare($count_sql);
        $count_stmt->execute([$member_numeric_id]);
        $count_result = $count_stmt->fetch(PDO::FETCH_ASSOC);
        
        $response = [
            'missed_contributions' => $results,
            'total' => (int)$count_result['total'],
            'limit' => $limit,
            'offset' => $offset
        ];
        
        echo json_encode($response);
    } catch (Exception $e) {
        throw $e;
    }
}

function getAllMissedContributions($limit, $offset) {
    global $pdo;
    
    try {
        // Get all missed contributions
        $sql = "SELECT mc.*, m.first_name, m.last_name, m.member_id 
                FROM missed_contributions mc
                JOIN members m ON mc.member_id = m.id
                ORDER BY mc.missed_date DESC
                LIMIT ? OFFSET ?";
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute([$limit, $offset]);
        $results = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Get total count for pagination
        $count_sql = "SELECT COUNT(*) as total FROM missed_contributions";
        $count_stmt = $pdo->prepare($count_sql);
        $count_stmt->execute();
        $count_result = $count_stmt->fetch(PDO::FETCH_ASSOC);
        
        $response = [
            'missed_contributions' => $results,
            'total' => (int)$count_result['total'],
            'limit' => $limit,
            'offset' => $offset
        ];
        
        echo json_encode($response);
    } catch (Exception $e) {
        throw $e;
    }
}

function recordMissedContribution($data) {
    global $pdo;
    
    try {
        // Required fields
        $required_fields = ['member_id', 'amount', 'missed_date'];
        foreach ($required_fields as $field) {
            if (!isset($data[$field])) {
                http_response_code(400);
                echo json_encode(['error' => "Missing required field: $field"]);
                return;
            }
        }
        
        // Get member's numeric ID
        $member_check_sql = "SELECT id FROM members WHERE member_id = ?";
        $member_stmt = $pdo->prepare($member_check_sql);
        $member_stmt->execute([$data['member_id']]);
        $member_result = $member_stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$member_result) {
            http_response_code(404);
            echo json_encode(['error' => 'Member not found']);
            return;
        }
        
        $member_numeric_id = $member_result['id'];
        
        // Check if this missed contribution already exists
        $check_sql = "SELECT id FROM missed_contributions 
                      WHERE member_id = ? AND missed_date = ?";
        $check_stmt = $pdo->prepare($check_sql);
        $check_stmt->execute([$member_numeric_id, $data['missed_date']]);
        $existing = $check_stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($existing) {
            http_response_code(409);
            echo json_encode(['error' => 'Missed contribution already recorded for this date']);
            return;
        }
        
        // Insert new missed contribution
        $sql = "INSERT INTO missed_contributions 
                (member_id, amount, missed_date, reason, status, created_at) 
                VALUES (?, ?, ?, ?, ?, NOW())";
        
        $stmt = $pdo->prepare($sql);
        $result = $stmt->execute([
            $member_numeric_id,
            $data['amount'],
            $data['missed_date'],
            isset($data['reason']) ? $data['reason'] : null,
            isset($data['status']) ? $data['status'] : 'Unpaid'
        ]);
        
        if ($result) {
            $new_id = $pdo->lastInsertId();
            
            // Get the inserted record
            $select_sql = "SELECT mc.*, m.first_name, m.last_name, m.member_id 
                           FROM missed_contributions mc
                           JOIN members m ON mc.member_id = m.id
                           WHERE mc.id = ?";
            $select_stmt = $pdo->prepare($select_sql);
            $select_stmt->execute([$new_id]);
            $record = $select_stmt->fetch(PDO::FETCH_ASSOC);
            
            http_response_code(201);
            echo json_encode([
                'message' => 'Missed contribution recorded successfully',
                'missed_contribution' => $record
            ]);
        } else {
            throw new Exception('Failed to record missed contribution');
        }
    } catch (Exception $e) {
        throw $e;
    }
}

function updateMissedContribution($data) {
    global $pdo;
    
    try {
        // Update missed contribution
        $sql = "UPDATE missed_contributions SET 
                amount = ?, 
                missed_date = ?, 
                reason = ?, 
                status = ?, 
                updated_at = NOW() 
                WHERE id = ?";
        
        $stmt = $pdo->prepare($sql);
        $result = $stmt->execute([
            isset($data['amount']) ? $data['amount'] : null,
            isset($data['missed_date']) ? $data['missed_date'] : null,
            isset($data['reason']) ? $data['reason'] : null,
            isset($data['status']) ? $data['status'] : null,
            $data['id']
        ]);
        
        if ($result) {
            // Get the updated record
            $select_sql = "SELECT mc.*, m.first_name, m.last_name, m.member_id 
                           FROM missed_contributions mc
                           JOIN members m ON mc.member_id = m.id
                           WHERE mc.id = ?";
            $select_stmt = $pdo->prepare($select_sql);
            $select_stmt->execute([$data['id']]);
            $record = $select_stmt->fetch(PDO::FETCH_ASSOC);
            
            echo json_encode([
                'message' => 'Missed contribution updated successfully',
                'missed_contribution' => $record
            ]);
        } else {
            throw new Exception('Failed to update missed contribution');
        }
    } catch (Exception $e) {
        throw $e;
    }
}

function deleteMissedContribution($id) {
    global $pdo;
    
    try {
        // Delete missed contribution
        $sql = "DELETE FROM missed_contributions WHERE id = ?";
        $stmt = $pdo->prepare($sql);
        $result = $stmt->execute([$id]);
        
        if ($result) {
            echo json_encode(['message' => 'Missed contribution deleted successfully']);
        } else {
            throw new Exception('Failed to delete missed contribution');
        }
    } catch (Exception $e) {
        throw $e;
    }
}
?>