<?php
// MFT GROUP Meetings API
// This API handles meetings and attendance records

// Enable CORS for frontend access
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Credentials: true");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS");
header("Access-Control-Allow-Headers: Origin, X-Requested-With, Content-Type, Accept, Authorization");
header("Content-Type: application/json; charset=UTF-8");

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Log all requests for debugging
error_log("Meetings API Request: " . $_SERVER['REQUEST_METHOD'] . " " . $_SERVER['REQUEST_URI']);

// Include database configuration
require_once dirname(__DIR__) . '/database/config.php';

// Get request method
$method = isset($_SERVER['REQUEST_METHOD']) ? $_SERVER['REQUEST_METHOD'] : 'GET';

try {
    switch ($method) {
        case 'GET':
            error_log("Handling GET request");
            handleGetRequest();
            break;
        case 'POST':
            error_log("Handling POST request");
            handlePostRequest();
            break;
        case 'PUT':
            error_log("Handling PUT request");
            handlePutRequest();
            break;
        case 'DELETE':
            error_log("Handling DELETE request");
            handleDeleteRequest();
            break;
        default:
            error_log("Method not allowed: " . $method);
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            break;
    }
} catch (Exception $e) {
    error_log("Exception in main handler: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['error' => 'Internal server error: ' . $e->getMessage()]);
}

function handleGetRequest() {
    error_log("handleGetRequest called");
    // Get query parameters
    $meeting_id = isset($_GET['meeting_id']) ? (int)$_GET['meeting_id'] : null;
    $member_id = isset($_GET['member_id']) ? $_GET['member_id'] : null;
    $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 10;
    $offset = isset($_GET['offset']) ? (int)$_GET['offset'] : 0;
    $status = isset($_GET['status']) ? $_GET['status'] : null;
    $attendance_record_id = isset($_GET['id']) ? (int)$_GET['id'] : null;
    
    error_log("GET parameters - meeting_id: " . var_export($meeting_id, true) . 
              ", member_id: " . var_export($member_id, true) . 
              ", status: " . var_export($status, true) . 
              ", attendance_record_id: " . var_export($attendance_record_id, true));
    
    if ($meeting_id) {
        // Get specific meeting details
        getMeetingDetails($meeting_id);
    } else if ($member_id) {
        // Get attendance records for a specific member
        getMemberAttendance($member_id, $limit, $offset, $status);
    } else if ($attendance_record_id) {
        // Get specific attendance record details
        getAttendanceRecordDetails($attendance_record_id);
    } else {
        // Get all meetings
        getAllMeetings($limit, $offset, $status);
    }
}

function handlePostRequest() {
    error_log("handlePostRequest called");
    // Create a new meeting or record attendance
    $input = json_decode(file_get_contents('php://input'), true);
    
    error_log("POST input: " . var_export($input, true));
    
    if (!$input) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid JSON data']);
        return;
    }
    
    if (isset($input['attendance'])) {
        // Record attendance
        recordAttendance($input);
    } else {
        // Create meeting
        createMeeting($input);
    }
}

function handlePutRequest() {
    error_log("handlePutRequest called");
    // Update a meeting or attendance record
    $input = json_decode(file_get_contents('php://input'), true);
    
    error_log("PUT input: " . var_export($input, true));
    
    if (!$input || !isset($input['id'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Missing required fields']);
        return;
    }
    
    if (isset($input['attendance_status'])) {
        // Update attendance
        updateAttendance($input);
    } else {
        // Update meeting
        updateMeeting($input);
    }
}

function handleDeleteRequest() {
    error_log("handleDeleteRequest called");
    // Delete a meeting
    $id = isset($_GET['id']) ? (int)$_GET['id'] : null;
    
    error_log("Delete request for ID: " . var_export($id, true));
    
    if (!$id) {
        error_log("Missing ID parameter in delete request");
        http_response_code(400);
        echo json_encode(['error' => 'Missing ID parameter']);
        return;
    }
    
    // Log the ID for debugging
    error_log("Received delete request for meeting ID: " . $id);
    
    deleteMeeting($id);
}

function getAllMeetings($limit, $offset, $status = null) {
    try {
        $pdo = getDatabaseConnection();
        
        // Build the query with optional status filter
        // Fixed the GROUP BY issue by specifying individual columns
        $sql = "SELECT m.id, m.title, m.description, m.meeting_date, m.duration, m.location, 
                       m.agenda, m.minutes, m.status, m.created_by, m.created_at, m.updated_at,
                       mem.first_name as created_by_name, mem.last_name as created_by_lastname,
                       COUNT(ma.id) as total_attendees,
                       SUM(CASE WHEN ma.attendance_status = 'Present' THEN 1 ELSE 0 END) as present_count
                FROM meetings m
                JOIN members mem ON m.created_by = mem.id
                LEFT JOIN meeting_attendance ma ON m.id = ma.meeting_id";
        
        $params = [];
        
        if ($status) {
            $sql .= " WHERE m.status = ?";
            $params[] = $status;
        }
        
        $sql .= " GROUP BY m.id, m.title, m.description, m.meeting_date, m.duration, m.location, 
                         m.agenda, m.minutes, m.status, m.created_by, m.created_at, m.updated_at,
                         mem.first_name, mem.last_name
                  ORDER BY m.meeting_date DESC 
                  LIMIT ? OFFSET ?";
        $params[] = $limit;
        $params[] = $offset;
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        $results = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Get total count for pagination
        $count_sql = "SELECT COUNT(*) as total FROM meetings";
        $count_params = [];
        
        if ($status) {
            $count_sql .= " WHERE status = ?";
            $count_params[] = $status;
        }
        
        $count_stmt = $pdo->prepare($count_sql);
        $count_stmt->execute($count_params);
        $count_result = $count_stmt->fetch(PDO::FETCH_ASSOC);
        
        $response = [
            'meetings' => $results,
            'total' => (int)$count_result['total'],
            'limit' => $limit,
            'offset' => $offset
        ];
        
        echo json_encode($response);
    } catch (Exception $e) {
        error_log("Error in getAllMeetings: " . $e->getMessage());
        throw new Exception("Database error: " . $e->getMessage());
    }
}

function getMeetingDetails($meeting_id) {
    try {
        $pdo = getDatabaseConnection();
        
        // Get meeting details
        $sql = "SELECT m.*, mem.first_name as created_by_name, mem.last_name as created_by_lastname
                FROM meetings m
                JOIN members mem ON m.created_by = mem.id
                WHERE m.id = ?";
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute([$meeting_id]);
        $meeting = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$meeting) {
            http_response_code(404);
            echo json_encode(['error' => 'Meeting not found']);
            return;
        }
        
        // Get attendance records for this meeting
        $attendance_sql = "SELECT ma.*, mem.first_name, mem.last_name, mem.member_id
                          FROM meeting_attendance ma
                          JOIN members mem ON ma.member_id = mem.id
                          WHERE ma.meeting_id = ?
                          ORDER BY mem.first_name, mem.last_name";
        
        $attendance_stmt = $pdo->prepare($attendance_sql);
        $attendance_stmt->execute([$meeting_id]);
        $attendance = $attendance_stmt->fetchAll(PDO::FETCH_ASSOC);
        
        $response = [
            'meeting' => $meeting,
            'attendance' => $attendance
        ];
        
        echo json_encode($response);
    } catch (Exception $e) {
        throw $e;
    }
}

function getAttendanceRecordDetails($attendance_record_id) {
    try {
        $pdo = getDatabaseConnection();
        
        // Get specific attendance record details
        $sql = "SELECT ma.*, m.title as meeting_title, m.meeting_date, m.status as meeting_status
                FROM meeting_attendance ma
                JOIN meetings m ON ma.meeting_id = m.id
                WHERE ma.id = ?";
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute([$attendance_record_id]);
        $attendance = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$attendance) {
            http_response_code(404);
            echo json_encode(['error' => 'Attendance record not found']);
            return;
        }
        
        $response = [
            'attendance' => $attendance
        ];
        
        echo json_encode($response);
    } catch (Exception $e) {
        throw $e;
    }
}

function getMemberAttendance($member_id, $limit, $offset, $status = null) {
    try {
        $pdo = getDatabaseConnection();
        
        // First, get the member's numeric ID
        $member_check_sql = "SELECT id FROM members WHERE member_id = ?";
        $member_stmt = $pdo->prepare($member_check_sql);
        $member_stmt->execute([$member_id]);
        $member_result = $member_stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$member_result) {
            http_response_code(404);
            echo json_encode(['error' => 'Member not found']);
            return;
        }
        
        $member_numeric_id = $member_result['id'];
        
        // Get attendance records for this member
        $sql = "SELECT ma.*, m.title as meeting_title, m.meeting_date, m.status as meeting_status
                FROM meeting_attendance ma
                JOIN meetings m ON ma.meeting_id = m.id
                WHERE ma.member_id = ?";
        
        $params = [$member_numeric_id];
        
        if ($status) {
            $sql .= " AND ma.attendance_status = ?";
            $params[] = $status;
        }
        
        $sql .= " ORDER BY m.meeting_date DESC LIMIT ? OFFSET ?";
        $params[] = $limit;
        $params[] = $offset;
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        $results = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Get total count for pagination
        $count_sql = "SELECT COUNT(*) as total 
                      FROM meeting_attendance ma
                      JOIN meetings m ON ma.meeting_id = m.id
                      WHERE ma.member_id = ?";
        $count_params = [$member_numeric_id];
        
        if ($status) {
            $count_sql .= " AND ma.attendance_status = ?";
            $count_params[] = $status;
        }
        
        $count_stmt = $pdo->prepare($count_sql);
        $count_stmt->execute($count_params);
        $count_result = $count_stmt->fetch(PDO::FETCH_ASSOC);
        
        // Get attendance statistics
        $stats_sql = "SELECT 
                        COUNT(*) as total_meetings,
                        SUM(CASE WHEN ma.attendance_status = 'Present' THEN 1 ELSE 0 END) as attended,
                        SUM(CASE WHEN ma.attendance_status = 'Absent' THEN 1 ELSE 0 END) as missed,
                        SUM(CASE WHEN ma.attendance_status = 'Late' THEN 1 ELSE 0 END) as late,
                        SUM(ma.fine_imposed) as total_fines,
                        SUM(CASE WHEN ma.fine_status = 'Paid' THEN ma.fine_imposed ELSE 0 END) as paid_fines
                      FROM meeting_attendance ma
                      WHERE ma.member_id = ?";
        
        $stats_stmt = $pdo->prepare($stats_sql);
        $stats_stmt->execute([$member_numeric_id]);
        $stats = $stats_stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($stats['total_meetings'] > 0) {
            $attendance_rate = round(($stats['attended'] / $stats['total_meetings']) * 100, 1);
        } else {
            $attendance_rate = 0;
        }
        
        $stats['attendance_rate'] = $attendance_rate;
        
        $response = [
            'attendance_records' => $results,
            'statistics' => $stats,
            'total' => (int)$count_result['total'],
            'limit' => $limit,
            'offset' => $offset
        ];
        
        echo json_encode($response);
    } catch (Exception $e) {
        throw $e;
    }
}

function createMeeting($data) {
    try {
        $pdo = getDatabaseConnection();
        
        // Required fields
        $required_fields = ['title', 'meeting_date', 'created_by'];
        foreach ($required_fields as $field) {
            if (!isset($data[$field])) {
                http_response_code(400);
                echo json_encode(['error' => "Missing required field: $field"]);
                return;
            }
        }
        
        // Insert meeting - using the actual database structure
        $sql = "INSERT INTO meetings 
                (title, description, meeting_date, duration, location, agenda, minutes, status, created_by) 
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)";
        
        $stmt = $pdo->prepare($sql);
        $result = $stmt->execute([
            $data['title'],
            isset($data['description']) ? $data['description'] : null,
            $data['meeting_date'],
            isset($data['duration']) ? $data['duration'] : null,
            isset($data['location']) ? $data['location'] : null,
            isset($data['agenda']) ? $data['agenda'] : null,
            isset($data['minutes']) ? $data['minutes'] : null,
            isset($data['status']) ? $data['status'] : 'Scheduled',
            $data['created_by']
        ]);
        
        if ($result) {
            $new_id = $pdo->lastInsertId();
            
            // Get the inserted record
            $select_sql = "SELECT m.*, mem.first_name as created_by_name, mem.last_name as created_by_lastname
                           FROM meetings m
                           JOIN members mem ON m.created_by = mem.id
                           WHERE m.id = ?";
            $select_stmt = $pdo->prepare($select_sql);
            $select_stmt->execute([$new_id]);
            $record = $select_stmt->fetch(PDO::FETCH_ASSOC);
            
            http_response_code(201);
            echo json_encode([
                'message' => 'Meeting created successfully',
                'meeting' => $record
            ]);
        } else {
            throw new Exception('Failed to create meeting');
        }
    } catch (Exception $e) {
        throw $e;
    }
}

function recordAttendance($data) {
    try {
        $pdo = getDatabaseConnection();
        
        // Required fields
        $required_fields = ['meeting_id', 'member_id', 'attendance_status'];
        foreach ($required_fields as $field) {
            if (!isset($data[$field])) {
                http_response_code(400);
                echo json_encode(['error' => "Missing required field: $field"]);
                return;
            }
        }
        
        // Check if attendance record already exists
        $check_sql = "SELECT id FROM meeting_attendance 
                      WHERE meeting_id = ? AND member_id = ?";
        $check_stmt = $pdo->prepare($check_sql);
        $check_stmt->execute([$data['meeting_id'], $data['member_id']]);
        $existing = $check_stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($existing) {
            http_response_code(409);
            echo json_encode(['error' => 'Attendance record already exists for this member and meeting']);
            return;
        }
        
        // Insert attendance record
        $sql = "INSERT INTO meeting_attendance 
                (meeting_id, member_id, attendance_status, reason, fine_imposed, fine_status) 
                VALUES (?, ?, ?, ?, ?, ?)";
        
        $stmt = $pdo->prepare($sql);
        $result = $stmt->execute([
            $data['meeting_id'],
            $data['member_id'],
            $data['attendance_status'],
            isset($data['reason']) ? $data['reason'] : null,
            isset($data['fine_imposed']) ? $data['fine_imposed'] : 0.00,
            isset($data['fine_status']) ? $data['fine_status'] : 'Pending'
        ]);
        
        if ($result) {
            $new_id = $pdo->lastInsertId();
            
            // Get the inserted record
            $select_sql = "SELECT ma.*, mem.first_name, mem.last_name, mem.member_id, m.title as meeting_title
                           FROM meeting_attendance ma
                           JOIN members mem ON ma.member_id = mem.id
                           JOIN meetings m ON ma.meeting_id = m.id
                           WHERE ma.id = ?";
            $select_stmt = $pdo->prepare($select_sql);
            $select_stmt->execute([$new_id]);
            $record = $select_stmt->fetch(PDO::FETCH_ASSOC);
            
            http_response_code(201);
            echo json_encode([
                'message' => 'Attendance recorded successfully',
                'attendance' => $record
            ]);
        } else {
            throw new Exception('Failed to record attendance');
        }
    } catch (Exception $e) {
        throw $e;
    }
}

function updateMeeting($data) {
    try {
        $pdo = getDatabaseConnection();
        
        // Update meeting
        $sql = "UPDATE meetings SET 
                title = ?, 
                description = ?, 
                meeting_date = ?, 
                start_time = ?, 
                end_time = ?, 
                location = ?, 
                status = ?,
                updated_at = NOW() 
                WHERE id = ?";
        
        $stmt = $pdo->prepare($sql);
        $result = $stmt->execute([
            isset($data['title']) ? $data['title'] : null,
            isset($data['description']) ? $data['description'] : null,
            isset($data['meeting_date']) ? $data['meeting_date'] : null,
            isset($data['start_time']) ? $data['start_time'] : null,
            isset($data['end_time']) ? $data['end_time'] : null,
            isset($data['location']) ? $data['location'] : null,
            isset($data['status']) ? $data['status'] : null,
            $data['id']
        ]);
        
        if ($result) {
            // Get the updated record
            $select_sql = "SELECT m.*, mem.first_name as created_by_name, mem.last_name as created_by_lastname
                           FROM meetings m
                           JOIN members mem ON m.created_by = mem.id
                           WHERE m.id = ?";
            $select_stmt = $pdo->prepare($select_sql);
            $select_stmt->execute([$data['id']]);
            $record = $select_stmt->fetch(PDO::FETCH_ASSOC);
            
            echo json_encode([
                'message' => 'Meeting updated successfully',
                'meeting' => $record
            ]);
        } else {
            throw new Exception('Failed to update meeting');
        }
    } catch (Exception $e) {
        throw $e;
    }
}

function updateAttendance($data) {
    try {
        $pdo = getDatabaseConnection();
        
        // Update attendance record
        $sql = "UPDATE meeting_attendance SET 
                attendance_status = ?, 
                reason = ?, 
                fine_imposed = ?, 
                fine_status = ?,
                updated_at = NOW() 
                WHERE id = ?";
        
        $stmt = $pdo->prepare($sql);
        $result = $stmt->execute([
            isset($data['attendance_status']) ? $data['attendance_status'] : null,
            isset($data['reason']) ? $data['reason'] : null,
            isset($data['fine_imposed']) ? $data['fine_imposed'] : null,
            isset($data['fine_status']) ? $data['fine_status'] : null,
            $data['id']
        ]);
        
        if ($result) {
            // Get the updated record
            $select_sql = "SELECT ma.*, mem.first_name, mem.last_name, mem.member_id, m.title as meeting_title
                           FROM meeting_attendance ma
                           JOIN members mem ON ma.member_id = mem.id
                           JOIN meetings m ON ma.meeting_id = m.id
                           WHERE ma.id = ?";
            $select_stmt = $pdo->prepare($select_sql);
            $select_stmt->execute([$data['id']]);
            $record = $select_stmt->fetch(PDO::FETCH_ASSOC);
            
            echo json_encode([
                'message' => 'Attendance updated successfully',
                'attendance' => $record
            ]);
        } else {
            throw new Exception('Failed to update attendance');
        }
    } catch (Exception $e) {
        throw $e;
    }
}

function deleteMeeting($id) {
    try {
        // Ensure ID is an integer
        $id = (int)$id;
        
        $pdo = getDatabaseConnection();
        
        // Log the ID we're trying to delete for debugging
        error_log("Attempting to delete meeting with ID: " . $id);
        
        // First check if the meeting exists
        $check_sql = "SELECT id, title FROM meetings WHERE id = ?";
        $check_stmt = $pdo->prepare($check_sql);
        $check_stmt->execute([$id]);
        $meeting = $check_stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$meeting) {
            error_log("Meeting not found with ID: " . $id);
            http_response_code(404);
            echo json_encode(['error' => 'Meeting not found with ID: ' . $id]);
            return;
        }
        
        error_log("Meeting found with ID: " . $id . ", Title: " . $meeting['title'] . ", proceeding with deletion");
        
        // Check for any foreign key constraints that might prevent deletion
        try {
            // Check if there are any attendance records for this meeting
            $attendance_check_sql = "SELECT COUNT(*) as count FROM meeting_attendance WHERE meeting_id = ?";
            $attendance_stmt = $pdo->prepare($attendance_check_sql);
            $attendance_stmt->execute([$id]);
            $attendance_count = $attendance_stmt->fetch(PDO::FETCH_ASSOC)['count'];
            
            error_log("Attendance records found for meeting ID " . $id . ": " . $attendance_count);
        } catch (Exception $e) {
            error_log("Error checking attendance records: " . $e->getMessage());
        }
        
        // Delete meeting (attendance records will be deleted automatically due to CASCADE)
        $sql = "DELETE FROM meetings WHERE id = ?";
        $stmt = $pdo->prepare($sql);
        $result = $stmt->execute([$id]);
        
        if ($result) {
            error_log("Meeting deleted successfully with ID: " . $id);
            echo json_encode(['message' => 'Meeting deleted successfully']);
        } else {
            error_log("Failed to delete meeting with ID: " . $id);
            http_response_code(500);
            echo json_encode(['error' => 'Failed to delete meeting with ID: ' . $id]);
        }
    } catch (Exception $e) {
        error_log("Exception in deleteMeeting: " . $e->getMessage());
        http_response_code(500);
        echo json_encode(['error' => 'Exception occurred while deleting meeting: ' . $e->getMessage()]);
    }
}
?>