<?php
// MFT GROUP Loan Types API
// This API provides loan type data for the frontend dashboards

// Enable CORS for frontend access
header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");

// Include database configuration
require_once __DIR__ . '/../database/config.php';

// Check if loan_types table exists and create it if it doesn't
try {
    $pdo = getDatabaseConnection();
    $stmt = $pdo->query("SHOW TABLES LIKE 'loan_types'");
    $tableExists = $stmt->fetch();
    
    if (!$tableExists) {
        createLoanTypesTable();
    }
} catch (Exception $e) {
    error_log("Error checking/creating loan_types table: " . $e->getMessage());
}

// Get request method
$method = $_SERVER['REQUEST_METHOD'];

try {
    switch ($method) {
        case 'GET':
            handleGetRequest();
            break;
        case 'POST':
            handlePostRequest();
            break;
        case 'PUT':
            handlePutRequest();
            break;
        case 'DELETE':
            handleDeleteRequest();
            break;
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            break;
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Internal server error: ' . $e->getMessage()]);
}

function handleGetRequest() {
    // Table creation is handled at the beginning of the script
    // No need to check again here
    
    $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 50;
    $offset = isset($_GET['offset']) ? (int)$_GET['offset'] : 0;
    
    try {
        $pdo = getDatabaseConnection();
        
        // Get total count
        $countStmt = $pdo->query("SELECT COUNT(*) as total FROM loan_types");
        $countResult = $countStmt->fetch();
        $total = $countResult['total'];
        
        // Get loan types with pagination
        $stmt = $pdo->prepare("SELECT * FROM loan_types ORDER BY created_at DESC LIMIT ? OFFSET ?");
        $stmt->execute([$limit, $offset]);
        $loanTypes = $stmt->fetchAll();
        
        http_response_code(200);
        echo json_encode([
            'count' => count($loanTypes),
            'total' => $total,
            'offset' => $offset,
            'limit' => $limit,
            'loan_types' => $loanTypes
        ]);
    } catch (Exception $e) {
        error_log("Error fetching loan types: " . $e->getMessage());
        http_response_code(500);
        echo json_encode(['error' => 'Failed to fetch loan types']);
    }
}

function handlePostRequest() {
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid JSON data']);
        return;
    }
    
    // Validate required fields
    $required_fields = ['name', 'interest_rate'];
    foreach ($required_fields as $field) {
        if (!isset($input[$field]) || empty($input[$field])) {
            http_response_code(400);
            echo json_encode(['error' => "Missing required field: $field"]);
            return;
        }
    }
    
    try {
        $pdo = getDatabaseConnection();
        
        // Insert loan type
        $stmt = $pdo->prepare("INSERT INTO loan_types (name, interest_rate, max_amount, max_term, description, is_active) VALUES (?, ?, ?, ?, ?, ?)");
        $stmt->execute([
            $input['name'],
            $input['interest_rate'],
            $input['max_amount'] ?? null,
            $input['max_term'] ?? null,
            $input['description'] ?? null,
            isset($input['is_active']) ? $input['is_active'] : 1
        ]);
        
        $loanTypeId = $pdo->lastInsertId();
        
        http_response_code(201);
        echo json_encode([
            'message' => 'Loan type created successfully',
            'loan_type_id' => $loanTypeId
        ]);
    } catch (Exception $e) {
        error_log("Error creating loan type: " . $e->getMessage());
        http_response_code(500);
        echo json_encode(['error' => 'Failed to create loan type']);
    }
}

function handlePutRequest() {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
}

function handleDeleteRequest() {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
}

function createLoanTypesTable() {
    try {
        $pdo = getDatabaseConnection();
        
        $sql = "CREATE TABLE IF NOT EXISTS loan_types (
            id INT PRIMARY KEY AUTO_INCREMENT,
            name VARCHAR(100) NOT NULL,
            interest_rate DECIMAL(5,2) NOT NULL,
            max_amount DECIMAL(15,2) NULL,
            max_term INT NULL,
            description TEXT,
            is_active BOOLEAN DEFAULT TRUE,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
        )";
        
        $pdo->exec($sql);
        
        // Check if we already have loan types
        $countStmt = $pdo->query("SELECT COUNT(*) as count FROM loan_types");
        $countResult = $countStmt->fetch();
        
        // Only insert default loan types if the table is empty
        if ($countResult['count'] == 0) {
            $stmt = $pdo->prepare("INSERT INTO loan_types (name, interest_rate, max_amount, max_term, description) VALUES 
                ('Emergency Loan', 12.00, 50000.00, 12, 'Short-term loan for emergencies'),
                ('Business Loan', 10.00, 200000.00, 24, 'Loan for business investments'),
                ('Education Loan', 8.00, 100000.00, 36, 'Loan for educational purposes'),
                ('Agriculture Loan', 9.00, 150000.00, 18, 'Loan for agricultural activities'),
                ('Personal Loan', 11.00, 75000.00, 24, 'General purpose personal loan')");
            $stmt->execute();
        }
        
        error_log("Loan types table created successfully");
    } catch (Exception $e) {
        error_log("Error creating loan types table: " . $e->getMessage());
        throw $e;
    }
}
?>