<?php
// MFT GROUP Initialize Default Settings API
// This API endpoint initializes default system settings

// Enable CORS for frontend access
header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Methods: POST");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");

// Include database configuration
require_once __DIR__ . '/../database/config.php';

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit;
}

try {
    // Default settings to insert (same as in init-default-settings.php)
    $default_settings = [
        // General Settings
        'orgName' => 'MFT GROUP',
        'orgAcronym' => 'MFT',
        'orgDescription' => 'Empowering our community through collective savings, mutual support, and sustainable financial growth since 2013.',
        'orgEmail' => 'info@mftselfhelpgroup.co.ke',
        'orgPhone' => '+254 720 327 807',
        'orgAddress' => 'Nairobi, Kenya',
        'currency' => 'KES',
        'timezone' => 'EAT',
        
        // Organization Settings
        'orgType' => 'chama',
        'chairperson' => 'John Mwangi',
        'secretary' => 'Mary Wanjiru',
        'treasurer' => 'Peter Kamau',
        'membersCount' => 38,
        'foundedDate' => '2013-05-15',
        
        // Membership Settings
        'membershipStatus' => 'open',
        'minContribution' => 500,
        'maxLoanMultiplier' => 3,
        'membershipFee' => 1000,
        'lateFee' => 100,
        'requireApproval' => true,
        'allowGuests' => false,
        'approvalType' => 'manual',
        'maxMembers' => 100,
        'renewalPeriod' => 365,
        'pendingStatus' => true,
        'activeStatus' => true,
        'suspendedStatus' => true,
        'terminatedStatus' => true,
        'gracePeriod' => 30,
        'membershipTiers' => 'single',
        'tierBenefits' => 'Regular members have standard voting rights and access to all group services.',
        
        // Loan Settings
        'loanStatus' => 'enabled',
        'interestRate' => 12,
        'savingsInterest' => 8,
        'dividendPercentage' => 15,
        'fineRate' => 5,
        'maxLoanTerm' => 12,
        'enableDividends' => true,
        'compoundInterest' => false,
        'maxLoanAmount' => 50000,
        'loanTerms' => '3,6,9,12',
        'lateInterest' => 15,
        'gracePeriodDays' => 7,
        'collateralPolicy' => 'required',
        'minSavings' => 5000,
        'overdue1Day' => true,
        'overdue3Days' => true,
        'overdue7Days' => true,
        'overdue14Days' => true,
        'savingsProducts' => 'regular',
        'savingsTypes' => 'Regular Savings,Emergency Fund,Fixed Deposit',
        'savingsInterestRates' => '8,6,10',
        'lockInPeriods' => '0,3,12',
        'minWithdrawal' => 1000,
        'withdrawalFrequency' => 'anytime',
        'scheduledContributions' => true,
        
        // Notification Settings
        'notificationEmail' => 'notifications@mftselfhelpgroup.co.ke',
        'emailMember' => true,
        'smsMember' => false,
        'pushMember' => false,
        'emailAdmin' => true,
        'smsAdmin' => false,
        'pushAdmin' => true,
        'contributionReminder' => '1day',
        'meetingReminder' => '3days',
        
        // Security Settings
        'sessionTimeout' => 30,
        'passwordExpiry' => 90,
        'ipWhitelist' => "192.168.1.0/24\n203.0.113.42",
        'loginAttempts' => 5,
        
        // Content Settings
        'aboutPage' => true,
        'projectsPage' => true,
        'galleryPage' => true,
        'downloadsPage' => true,
        'bankingPage' => true,
        'menuOrder' => 'Home,About Us,Projects,Membership,Gallery,Downloads,Banking Details,Contact Us',
        'maxGalleryImages' => 50,
        'maxFileSize' => 5,
        'allowedFileTypes' => 'pdf,doc,docx,xls,xlsx,jpg,png',
        'maxDownloadSize' => 10,
        
        // Officials Settings
        'officialsDisplay' => 'all',
        'officialsOrder' => 'Chairperson,Secretary,Treasurer,Vice Chairperson,Committee Members',
        'officialRoles' => 'Chairperson,Secretary,Treasurer,Vice Chairperson,Committee Member',
        'termLength' => 24,
        'termLimits' => true,
        'transitionPeriod' => 30,
        'electionNotice' => true,
        
        // Banking Settings
        'defaultCurrency' => 'KES',
        'primaryAccount' => 'account1',
        'accountVisibility' => 'members',
        'enableReconciliation' => true,
        'reconciliationThreshold' => 1000,
        
        // Documentation Settings
        'docStorageLocation' => 'local',
        'storageQuota' => 1000,
        'enableVersioning' => true,
        'maxVersions' => 10,
        'archivePeriod' => 12,
        'deletePeriod' => 60,
        
        // Roles Settings
        'adminRoles' => 'Super Admin,Financial Admin,Content Admin,Viewer',
        'defaultRole' => 'super',
        'manageLoans' => true,
        'manageMembers' => true,
        'manageContent' => true,
        'viewReports' => true,
        'disableAccounts' => true,
        'deleteAccounts' => false,
        
        // System Settings
        'backupFrequency' => 'daily',
        'backupRetention' => 30,
        'maintenanceMode' => false,
        'maintenanceMessage' => 'The system is currently under maintenance. We\'ll be back shortly.',
        'auditTrail' => true,
        'logRetention' => 90,
        
        // Integration Settings
        'mpesaStatus' => 'enabled',
        'mpesaKey' => 'sk_live_xxxxxxxxxxxxxxxx',
        'mpesaSecret' => 'sk_live_xxxxxxxxxxxxxxxx',
        'smsProvider' => 'twilio',
        'smsApiKey' => 'sk_live_xxxxxxxxxxxxxxxx',
        'smsSender' => 'MFTGROUP',
        'smtpHost' => 'smtp.gmail.com',
        'smtpPort' => 587,
        'smtpUser' => 'notifications@mftselfhelpgroup.co.ke',
        'smtpPass' => 'xxxxxxxxxxxxxxxx',
        
        // Branding Settings
        'primaryColor' => '#FF5C1B',
        'secondaryColor' => '#001a33',
        'themeMode' => 'dark',
        'fontFamily' => 'inter'
    ];

    // Initialize counter for inserted settings
    $inserted_count = 0;
    
    // Begin transaction
    beginTransaction();
    
    // Insert each setting
    foreach ($default_settings as $key => $value) {
        // Convert boolean values to strings for database storage
        if (is_bool($value)) {
            $value = $value ? 'true' : 'false';
        } elseif (is_array($value)) {
            $value = json_encode($value);
        }
        
        // Check if setting already exists
        $check_sql = "SELECT id FROM system_settings WHERE setting_key = ?";
        $existing = fetchSingleRow($check_sql, [$key]);
        
        if (!$existing) {
            // Insert setting
            $sql = "INSERT INTO system_settings (setting_key, setting_value) VALUES (?, ?)";
            executeNonQuery($sql, [$key, $value]);
            $inserted_count++;
        }
    }
    
    // Commit transaction
    commitTransaction();
    
    http_response_code(200);
    echo json_encode([
        'status' => 'success',
        'message' => "Successfully initialized $inserted_count settings.",
        'settings_count' => $inserted_count
    ]);
    
} catch (Exception $e) {
    // Rollback transaction on error
    try {
        rollbackTransaction();
    } catch (Exception $rollbackException) {
        // Ignore rollback errors
    }
    
    http_response_code(500);
    echo json_encode([
        'status' => 'error',
        'error' => 'Failed to initialize settings: ' . $e->getMessage()
    ]);
}
?>