<?php
// MFT GROUP Fines Payment API
// This API handles recording of fines payments

// Enable CORS for frontend access
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Credentials: true");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS");
header("Access-Control-Allow-Headers: Origin, X-Requested-With, Content-Type, Accept, Authorization");
header("Content-Type: application/json; charset=UTF-8");

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Include database configuration
require_once dirname(__DIR__) . '/database/config.php';

// Initialize database connection
global $pdo;
$pdo = getDatabaseConnection();

// Get request method
$method = $_SERVER['REQUEST_METHOD'];

try {
    switch ($method) {
        case 'POST':
            handlePostRequest();
            break;
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            break;
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Internal server error: ' . $e->getMessage()]);
}

function handlePostRequest() {
    global $pdo;
    
    // Record fines payment
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid JSON data']);
        return;
    }
    
    // Required fields
    $required_fields = ['member_id', 'amount', 'payment_reference'];
    foreach ($required_fields as $field) {
        if (!isset($input[$field])) {
            http_response_code(400);
            echo json_encode(['error' => "Missing required field: $field"]);
            return;
        }
    }
    
    // Validate amount
    if (!is_numeric($input['amount']) || $input['amount'] <= 0) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid amount. Must be a positive number']);
        return;
    }
    
    try {
        // Get member's numeric ID
        $member_check_sql = "SELECT id FROM members WHERE member_id = ?";
        $member_stmt = $pdo->prepare($member_check_sql);
        $member_stmt->execute([$input['member_id']]);
        $member_result = $member_stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$member_result) {
            http_response_code(404);
            echo json_encode(['error' => 'Member not found']);
            return;
        }
        
        $member_numeric_id = $member_result['id'];
        
        // Start transaction
        $pdo->beginTransaction();
        
        // Update existing unpaid fines for this member
        $fines_sql = "SELECT * FROM fines WHERE member_id = ? AND status = 'Unpaid' ORDER BY fine_date ASC";
        $fines_stmt = $pdo->prepare($fines_sql);
        $fines_stmt->execute([$member_numeric_id]);
        $unpaid_fines = $fines_stmt->fetchAll(PDO::FETCH_ASSOC);
        
        $remaining_amount = $input['amount'];
        $paid_fines = [];
        
        foreach ($unpaid_fines as $fine) {
            if ($remaining_amount <= 0) {
                break;
            }
            
            if ($remaining_amount >= $fine['amount']) {
                // Fully pay this fine
                $update_sql = "UPDATE fines SET status = 'Paid', payment_reference = ?, paid_date = ? WHERE id = ?";
                $update_stmt = $pdo->prepare($update_sql);
                $update_stmt->execute([
                    $input['payment_reference'],
                    date('Y-m-d'),
                    $fine['id']
                ]);
                
                $remaining_amount -= $fine['amount'];
                $paid_fines[] = $fine['id'];
            } else {
                // Partially pay this fine (in a real implementation, you might want to handle this differently)
                // For now, we'll leave it as unpaid
                break;
            }
        }
        
        // If there's still remaining amount and no fines were paid, create a new fine record
        if ($remaining_amount > 0 && empty($paid_fines)) {
            $insert_sql = "INSERT INTO fines (member_id, amount, reason, description, status, payment_reference, fine_date, created_by) 
                           VALUES (?, ?, 'Other', ?, 'Paid', ?, ?, ?)";
            $insert_stmt = $pdo->prepare($insert_sql);
            $insert_stmt->execute([
                $member_numeric_id,
                $remaining_amount,
                'Overpayment adjustment',
                $input['payment_reference'],
                date('Y-m-d'),
                $member_numeric_id // Created by the member themselves
            ]);
        }
        
        $pdo->commit();
        
        // Return success response
        http_response_code(200);
        echo json_encode([
            'success' => true,
            'message' => 'Fines payment recorded successfully',
            'paid_fines' => $paid_fines,
            'remaining_amount' => $remaining_amount
        ]);
    } catch (Exception $e) {
        if ($pdo->inTransaction()) {
            $pdo->rollback();
        }
        throw $e;
    }
}
?>