<?php
/**
 * Unified Export API for MFT GROUP
 * Handles export requests for various data types in PDF or Excel formats
 */

// Enable CORS for frontend access
header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Methods: POST");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");

// Include required files
require_once __DIR__ . '/../database/config.php';
require_once __DIR__ . '/dompdf-utils.php';

// Try to include PhpSpreadsheet for Excel export
$autoloadPath = __DIR__ . '/../vendor/autoload.php';
if (file_exists($autoloadPath)) {
    require_once $autoloadPath;
}

/**
 * Handle export requests
 */
function handleExportRequest() {
    // Handle both JSON and form data submissions
    $input = [];
    $content_type = isset($_SERVER['CONTENT_TYPE']) ? $_SERVER['CONTENT_TYPE'] : '';
    
    if (strpos($content_type, 'application/json') !== false) {
        // JSON data
        $input = json_decode(file_get_contents('php://input'), true);
        if (!$input) {
            http_response_code(400);
            echo json_encode(['error' => 'Invalid JSON data']);
            return;
        }
    } else {
        // Form data
        $input = [
            'export_type' => isset($_POST['export_type']) ? $_POST['export_type'] : '',
            'format' => isset($_POST['format']) ? strtolower($_POST['format']) : 'pdf',
            'data' => isset($_POST['data']) ? json_decode($_POST['data'], true) : [],
            'filename' => isset($_POST['filename']) ? $_POST['filename'] : 'export_' . date('Y-m-d_H-i-s'),
            'title' => isset($_POST['title']) ? $_POST['title'] : 'MFT GROUP Export'
        ];
    }
    
    $export_type = $input['export_type'];
    $format = $input['format'];
    $data = $input['data'];
    $filename = $input['filename'];
    $title = $input['title'];
    
    // Debug: Log the received data
    error_log('Export request received: export_type=' . $export_type . ', format=' . $format);
    
    try {
        switch ($export_type) {
            case 'members':
                exportMembers($format, $filename, $title);
                break;
            case 'loans':
                exportLoans($format, $filename, $title, $data);
                break;
            case 'contributions':
                exportContributions($format, $filename, $title, $data);
                break;
            case 'fines':
                exportFines($format, $filename, $title, $data);
                break;
            case 'projects':
                exportProjects($format, $filename, $title, $data);
                break;
            case 'member':
                exportMemberReport($format, $filename, $title, $data);
                break;
            case 'generic':
                exportGenericData($format, $filename, $title, $data);
                break;
            default:
                // Debug: Log the invalid export type
                error_log('Invalid export type received: ' . $export_type);
                http_response_code(400);
                echo json_encode(['error' => 'Invalid export type: ' . $export_type]);
                return;
        }
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Export failed: ' . $e->getMessage()]);
    }
}

/**
 * Export members data
 */
function exportMembers($format, $filename, $title) {
    try {
        // Get database connection
        $pdo = getDatabaseConnection();
        
        // Fetch all members data
        $stmt = $pdo->query("SELECT * FROM members ORDER BY last_name, first_name");
        $members = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        if (empty($members)) {
            throw new Exception("No members found in the database.");
        }
        
        if ($format === 'excel' || $format === 'xlsx') {
            exportMembersToExcel($members, $filename);
        } else {
            exportMembersToPdf($members, $filename, $title);
        }
    } catch (Exception $e) {
        throw new Exception("Failed to export members: " . $e->getMessage());
    }
}

/**
 * Export members to Excel
 */
function exportMembersToExcel($members, $filename) {
    try {
        // Check if PhpSpreadsheet is available
        if (!class_exists('\PhpOffice\PhpSpreadsheet\Spreadsheet')) {
            throw new Exception("PhpSpreadsheet library not available for Excel export");
        }
        
        // Create new Spreadsheet object
        $spreadsheet = new \PhpOffice\PhpSpreadsheet\Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();
        
        // Set document properties
        $spreadsheet->getProperties()
            ->setCreator("MFT SELF HELP GROUP")
            ->setLastModifiedBy("MFT SELF HELP GROUP")
            ->setTitle("Members Export")
            ->setSubject("Members Data")
            ->setDescription("Export of all members data from MFT SELF HELP GROUP system");
        
        // Add headers
        $headers = array_keys($members[0]);
        $column = 'A';
        foreach ($headers as $header) {
            $sheet->setCellValue($column . '1', ucfirst(str_replace('_', ' ', $header)));
            $column++;
        }
        
        // Add data
        $row = 2;
        foreach ($members as $member) {
            $column = 'A';
            foreach ($member as $value) {
                $sheet->setCellValue($column . $row, $value);
                $column++;
            }
            $row++;
        }
        
        // Format headers
        $headerRange = 'A1:' . $sheet->getHighestColumn() . '1';
        $sheet->getStyle($headerRange)->applyFromArray([
            'font' => [
                'bold' => true,
                'color' => ['rgb' => 'FFFFFF']
            ],
            'fill' => [
                'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
                'startColor' => ['rgb' => '3498db']
            ],
            'alignment' => [
                'horizontal' => \PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_CENTER
            ]
        ]);
        
        // Add borders to all cells
        $dataRange = 'A1:' . $sheet->getHighestColumn() . $sheet->getHighestRow();
        $sheet->getStyle($dataRange)->applyFromArray([
            'borders' => [
                'allBorders' => [
                    'borderStyle' => \PhpOffice\PhpSpreadsheet\Style\Border::BORDER_THIN,
                    'color' => ['rgb' => '000000']
                ]
            ]
        ]);
        
        // Auto-size columns
        foreach (range('A', $sheet->getHighestColumn()) as $col) {
            $sheet->getColumnDimension($col)->setAutoSize(true);
        }
        
        // Freeze the header row
        $sheet->freezePane('A2');
        
        // Set headers for Excel download
        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header('Content-Disposition: attachment;filename="' . $filename . '.xlsx"');
        header('Cache-Control: max-age=0');
        
        // Create Excel file and send to browser
        $writer = new \PhpOffice\PhpSpreadsheet\Writer\Xlsx($spreadsheet);
        $writer->save('php://output');
        
        exit;
    } catch (Exception $e) {
        throw new Exception("Failed to export members to Excel: " . $e->getMessage());
    }
}

/**
 * Export members to PDF
 */
function exportMembersToPdf($members, $filename, $title) {
    try {
        // Generate HTML for the PDF
        $html = generateMembersHtml($members, $title);
        
        // Generate PDF using Dompdf
        generatePdfWithDompdf($html, $filename, [], 'D');
    } catch (Exception $e) {
        throw new Exception("Failed to export members to PDF: " . $e->getMessage());
    }
}

/**
 * Generate HTML for members export
 */
function generateMembersHtml($members, $title) {
    $html = '
    <!DOCTYPE html>
    <html>
    <head>
        <meta charset="UTF-8">
        <title>' . htmlspecialchars($title) . '</title>
        <style>
            body { font-family: Arial, sans-serif; font-size: 10pt; }
            .header { text-align: center; margin-bottom: 20px; }
            .header h1 { color: #14375d; margin: 0; }
            table { width: 100%; border-collapse: collapse; margin: 20px 0; }
            th, td { border: 1px solid #ddd; padding: 8px; text-align: left; }
            th { background-color: #f2f2f2; }
            .footer { margin-top: 30px; font-size: 9pt; color: #666; text-align: center; }
        </style>
    </head>
    <body>
        <div class="header">
            <h1>' . htmlspecialchars($title) . '</h1>
        </div>
        <table>
            <thead>
                <tr>';
    
    // Generate table headers from first row keys if data is not empty
    if (!empty($members)) {
        $first_row = reset($members);
        foreach (array_keys($first_row) as $header) {
            $html .= '<th>' . htmlspecialchars(ucfirst(str_replace('_', ' ', $header))) . '</th>';
        }
        $html .= '</tr>
            </thead>
            <tbody>';
        
        // Generate table rows
        foreach ($members as $row) {
            $html .= '<tr>';
            foreach ($row as $cell) {
                $html .= '<td>' . htmlspecialchars($cell) . '</td>';
            }
            $html .= '</tr>';
        }
    }
    
    $html .= '
            </tbody>
        </table>
        <div class="footer">
            Generated on ' . date('F j, Y') . ' by MFT GROUP System
        </div>
    </body>
    </html>';
    
    return $html;
}

/**
 * Export loans data
 */
function exportLoans($format, $filename, $title, $data) {
    try {
        if ($format === 'excel' || $format === 'xlsx') {
            exportGenericToExcel($data, $filename, $title);
        } else {
            exportGenericToPdf($data, $filename, $title);
        }
    } catch (Exception $e) {
        throw new Exception("Failed to export loans: " . $e->getMessage());
    }
}

/**
 * Export contributions data
 */
function exportContributions($format, $filename, $title, $data) {
    try {
        if ($format === 'excel' || $format === 'xlsx') {
            exportGenericToExcel($data, $filename, $title);
        } else {
            exportGenericToPdf($data, $filename, $title);
        }
    } catch (Exception $e) {
        throw new Exception("Failed to export contributions: " . $e->getMessage());
    }
}

/**
 * Export fines data
 */
function exportFines($format, $filename, $title, $data) {
    try {
        if ($format === 'excel' || $format === 'xlsx') {
            exportGenericToExcel($data, $filename, $title);
        } else {
            exportGenericToPdf($data, $filename, $title);
        }
    } catch (Exception $e) {
        throw new Exception("Failed to export fines: " . $e->getMessage());
    }
}

/**
 * Export projects data
 */
function exportProjects($format, $filename, $title, $data) {
    try {
        if ($format === 'excel' || $format === 'xlsx') {
            exportGenericToExcel($data, $filename, $title);
        } else {
            exportGenericToPdf($data, $filename, $title);
        }
    } catch (Exception $e) {
        throw new Exception("Failed to export projects: " . $e->getMessage());
    }
}

/**
 * Export individual member report
 */
function exportMemberReport($format, $filename, $title, $data) {
    try {
        if ($format === 'excel' || $format === 'xlsx') {
            exportMemberReportToExcel($data, $filename, $title);
        } else {
            exportMemberReportToPdf($data, $filename, $title);
        }
    } catch (Exception $e) {
        throw new Exception("Failed to export member report: " . $e->getMessage());
    }
}

/**
 * Export generic data
 */
function exportGenericData($format, $filename, $title, $data) {
    try {
        if ($format === 'excel' || $format === 'xlsx') {
            exportGenericToExcel($data, $filename, $title);
        } else {
            exportGenericToPdf($data, $filename, $title);
        }
    } catch (Exception $e) {
        throw new Exception("Failed to export data: " . $e->getMessage());
    }
}

/**
 * Export generic data to Excel
 */
function exportGenericToExcel($data, $filename, $title) {
    try {
        // Check if PhpSpreadsheet is available
        if (!class_exists('\PhpOffice\PhpSpreadsheet\Spreadsheet')) {
            throw new Exception("PhpSpreadsheet library not available for Excel export");
        }
        
        // Create new Spreadsheet object
        $spreadsheet = new \PhpOffice\PhpSpreadsheet\Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();
        
        // Set document properties
        $spreadsheet->getProperties()
            ->setCreator("MFT GROUP")
            ->setLastModifiedBy("MFT GROUP")
            ->setTitle($title)
            ->setSubject($title)
            ->setDescription("Export from MFT GROUP system");
        
        // Add headers
        if (!empty($data)) {
            $headers = array_keys($data[0]);
            $column = 'A';
            foreach ($headers as $header) {
                $sheet->setCellValue($column . '1', ucfirst(str_replace('_', ' ', $header)));
                $column++;
            }
            
            // Add data
            $row = 2;
            foreach ($data as $rowData) {
                $column = 'A';
                foreach ($rowData as $value) {
                    $sheet->setCellValue($column . $row, $value);
                    $column++;
                }
                $row++;
            }
            
            // Format headers
            $headerRange = 'A1:' . $sheet->getHighestColumn() . '1';
            $sheet->getStyle($headerRange)->applyFromArray([
                'font' => [
                    'bold' => true,
                    'color' => ['rgb' => 'FFFFFF']
                ],
                'fill' => [
                    'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
                    'startColor' => ['rgb' => '3498db']
                ],
                'alignment' => [
                    'horizontal' => \PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_CENTER
                ]
            ]);
            
            // Add borders to all cells
            $dataRange = 'A1:' . $sheet->getHighestColumn() . $sheet->getHighestRow();
            $sheet->getStyle($dataRange)->applyFromArray([
                'borders' => [
                    'allBorders' => [
                        'borderStyle' => \PhpOffice\PhpSpreadsheet\Style\Border::BORDER_THIN,
                        'color' => ['rgb' => '000000']
                    ]
                ]
            ]);
            
            // Auto-size columns
            foreach (range('A', $sheet->getHighestColumn()) as $col) {
                $sheet->getColumnDimension($col)->setAutoSize(true);
            }
            
            // Freeze the header row
            $sheet->freezePane('A2');
        }
        
        // Set headers for Excel download
        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header('Content-Disposition: attachment;filename="' . $filename . '.xlsx"');
        header('Cache-Control: max-age=0');
        
        // Create Excel file and send to browser
        $writer = new \PhpOffice\PhpSpreadsheet\Writer\Xlsx($spreadsheet);
        $writer->save('php://output');
        
        exit;
    } catch (Exception $e) {
        throw new Exception("Failed to export to Excel: " . $e->getMessage());
    }
}

/**
 * Export member report to Excel
 */
function exportMemberReportToExcel($data, $filename, $title) {
    try {
        // Check if PhpSpreadsheet is available
        if (!class_exists('\PhpOffice\PhpSpreadsheet\Spreadsheet')) {
            throw new Exception("PhpSpreadsheet library not available for Excel export");
        }
        
        // Create new Spreadsheet object
        $spreadsheet = new \PhpOffice\PhpSpreadsheet\Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();
        
        // Set document properties
        $spreadsheet->getProperties()
            ->setCreator("MFT GROUP")
            ->setLastModifiedBy("MFT GROUP")
            ->setTitle($title)
            ->setSubject($title)
            ->setDescription("Member Report from MFT GROUP system");
        
        // Add title
        $sheet->setCellValue('A1', $title);
        $sheet->mergeCells('A1:B1');
        $sheet->getStyle('A1')->getFont()->setBold(true)->setSize(14);
        
        // Add data
        $row = 3;
        foreach ($data as $item) {
            // Check if this is a section header (empty field value)
            if (empty($item['field'])) {
                // Empty line or section separator
                $row++;
                continue;
            }
            
            // Check if this is a section title (empty value)
            if (empty($item['value'])) {
                // Section title
                $sheet->setCellValue('A' . $row, $item['field']);
                $sheet->getStyle('A' . $row)->getFont()->setBold(true);
                $sheet->mergeCells('A' . $row . ':B' . $row);
            } else {
                // Regular data row
                $sheet->setCellValue('A' . $row, $item['field']);
                $sheet->setCellValue('B' . $row, $item['value']);
            }
            $row++;
        }
        
        // Auto-size columns
        $sheet->getColumnDimension('A')->setAutoSize(true);
        $sheet->getColumnDimension('B')->setAutoSize(true);
        
        // Set headers for Excel download
        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header('Content-Disposition: attachment;filename="' . $filename . '.xlsx"');
        header('Cache-Control: max-age=0');
        
        // Create Excel file and send to browser
        $writer = new \PhpOffice\PhpSpreadsheet\Writer\Xlsx($spreadsheet);
        $writer->save('php://output');
        
        exit;
    } catch (Exception $e) {
        throw new Exception("Failed to export member report to Excel: " . $e->getMessage());
    }
}

/**
 * Export member report to PDF
 */
function exportMemberReportToPdf($data, $filename, $title) {
    try {
        // Generate HTML for the PDF
        $html = generateMemberReportHtml($data, $title);
        
        // Generate PDF using Dompdf
        generatePdfWithDompdf($html, $filename, [], 'D');
    } catch (Exception $e) {
        throw new Exception("Failed to export member report to PDF: " . $e->getMessage());
    }
}

/**
 * Generate HTML for member report
 */
function generateMemberReportHtml($data, $title) {
    $html = '
    <!DOCTYPE html>
    <html>
    <head>
        <meta charset="UTF-8">
        <title>' . htmlspecialchars($title) . '</title>
        <style>
            body { font-family: Arial, sans-serif; font-size: 10pt; }
            .header { text-align: center; margin-bottom: 20px; }
            .header h1 { color: #14375d; margin: 0; }
            table { width: 100%; border-collapse: collapse; margin: 20px 0; }
            th, td { border: 1px solid #ddd; padding: 8px; text-align: left; }
            th { background-color: #f2f2f2; }
            .section-title { font-weight: bold; background-color: #e6f3ff; }
            .footer { margin-top: 30px; font-size: 9pt; color: #666; text-align: center; }
        </style>
    </head>
    <body>
        <div class="header">
            <h1>' . htmlspecialchars($title) . '</h1>
        </div>
        <table>
            <tbody>';
    
    foreach ($data as $item) {
        // Check if this is a section header (empty field value)
        if (empty($item['field'])) {
            // Empty line or section separator
            $html .= '<tr><td colspan="2">&nbsp;</td></tr>';
        } elseif (empty($item['value'])) {
            // Section title
            $html .= '<tr class="section-title"><td colspan="2">' . htmlspecialchars($item['field']) . '</td></tr>';
        } else {
            // Regular data row
            $html .= '<tr><td>' . htmlspecialchars($item['field']) . '</td><td>' . htmlspecialchars($item['value']) . '</td></tr>';
        }
    }
    
    $html .= '
            </tbody>
        </table>
        <div class="footer">
            Generated on ' . date('F j, Y') . ' by MFT GROUP System
        </div>
    </body>
    </html>';
    
    return $html;
}

/**
 * Export generic data to PDF
 */
function exportGenericToPdf($data, $filename, $title) {
    try {
        // Generate HTML for the PDF
        $html = generateGenericHtml($data, $title);
        
        // Generate PDF using Dompdf
        generatePdfWithDompdf($html, $filename, [], 'D');
    } catch (Exception $e) {
        throw new Exception("Failed to export to PDF: " . $e->getMessage());
    }
}

/**
 * Generate HTML for generic data export
 */
function generateGenericHtml($data, $title) {
    $html = '
    <!DOCTYPE html>
    <html>
    <head>
        <meta charset="UTF-8">
        <title>' . htmlspecialchars($title) . '</title>
        <style>
            body { font-family: Arial, sans-serif; font-size: 10pt; }
            .header { text-align: center; margin-bottom: 20px; }
            .header h1 { color: #14375d; margin: 0; }
            table { width: 100%; border-collapse: collapse; margin: 20px 0; }
            th, td { border: 1px solid #ddd; padding: 8px; text-align: left; }
            th { background-color: #f2f2f2; }
            .footer { margin-top: 30px; font-size: 9pt; color: #666; text-align: center; }
        </style>
    </head>
    <body>
        <div class="header">
            <h1>' . htmlspecialchars($title) . '</h1>
        </div>
        <table>
            <thead>
                <tr>';
    
    // Generate table headers from first row keys if data is not empty
    if (!empty($data)) {
        $first_row = reset($data);
        foreach (array_keys($first_row) as $header) {
            $html .= '<th>' . htmlspecialchars(ucfirst(str_replace('_', ' ', $header))) . '</th>';
        }
        $html .= '</tr>
            </thead>
            <tbody>';
        
        // Generate table rows
        foreach ($data as $row) {
            $html .= '<tr>';
            foreach ($row as $cell) {
                $html .= '<td>' . htmlspecialchars($cell) . '</td>';
            }
            $html .= '</tr>';
        }
    }
    
    $html .= '
            </tbody>
        </table>
        <div class="footer">
            Generated on ' . date('F j, Y') . ' by MFT GROUP System
        </div>
    </body>
    </html>';
    
    return $html;
}

// Handle the request
$method = $_SERVER['REQUEST_METHOD'];

try {
    switch ($method) {
        case 'POST':
            handleExportRequest();
            break;
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            break;
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Internal server error: ' . $e->getMessage()]);
}
?>