<?php
// MFT GROUP Documents API
// This API provides document data for the frontend dashboards

// Enable CORS for frontend access
header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");

// Include database configuration
require_once __DIR__ . '/../database/config.php';

// Get request method
$method = $_SERVER['REQUEST_METHOD'];

try {
    switch ($method) {
        case 'GET':
            handleGetRequest();
            break;
        case 'POST':
            handlePostRequest();
            break;
        case 'PUT':
            handlePutRequest();
            break;
        case 'DELETE':
            handleDeleteRequest();
            break;
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            break;
    }
} catch (Exception $e) {
    error_log("Documents API Error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['error' => 'Internal server error: ' . $e->getMessage()]);
}

function handleGetRequest() {
    if (isset($_GET['id'])) {
        // Get specific document
        getDocument($_GET['id']);
    } else if (isset($_GET['member_id'])) {
        // Get documents for a specific member
        getMemberDocuments($_GET['member_id']);
    } else if (isset($_GET['category'])) {
        // Get documents by category
        getDocumentsByCategory($_GET['category']);
    } else {
        // Get all documents with optional filters
        getDocuments();
    }
}

function getDocument($id) {
    try {
        $sql = "SELECT d.id, d.title, d.description, d.file_path, d.file_type, d.file_size,
                       d.category, d.uploaded_by, d.related_member_id, d.related_meeting_id,
                       d.is_public, d.status, d.created_at, d.updated_at,
                       m.first_name, m.last_name, m.member_id as member_ref,
                       rm.first_name as related_member_first_name, rm.last_name as related_member_last_name,
                       rm.member_id as related_member_ref
                FROM documents d
                LEFT JOIN members m ON d.uploaded_by = m.id
                LEFT JOIN members rm ON d.related_member_id = rm.id
                WHERE d.id = ?";
        
        $document = fetchSingleRow($sql, [$id]);
        
        if ($document) {
            http_response_code(200);
            echo json_encode($document);
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Document not found']);
        }
    } catch (Exception $e) {
        error_log("Error fetching document: " . $e->getMessage());
        http_response_code(500);
        echo json_encode(['error' => 'Error fetching document: ' . $e->getMessage()]);
    }
}

function getMemberDocuments($member_id) {
    try {
        // Validate member_id
        if (empty($member_id)) {
            http_response_code(400);
            echo json_encode(['error' => 'Member ID is required']);
            return;
        }
        
        // First, we need to get the numeric member ID from the member_id string
        $memberSql = "SELECT id FROM members WHERE member_id = ?";
        $memberResult = fetchSingleRow($memberSql, [$member_id]);
        
        if (!$memberResult) {
            http_response_code(404);
            echo json_encode(['error' => 'Member not found']);
            return;
        }
        
        $numericMemberId = $memberResult['id'];
        
        $sql = "SELECT d.id, d.title, d.description, d.file_path, d.file_type, d.file_size,
                       d.category, d.uploaded_by, d.related_member_id, d.related_meeting_id,
                       d.is_public, d.status, d.created_at, d.updated_at,
                       m.first_name, m.last_name, m.member_id as member_ref,
                       rm.first_name as related_member_first_name, rm.last_name as related_member_last_name,
                       rm.member_id as related_member_ref
                FROM documents d
                LEFT JOIN members m ON d.uploaded_by = m.id
                LEFT JOIN members rm ON d.related_member_id = rm.id
                WHERE d.related_member_id = ?
                ORDER BY d.created_at DESC";
        
        $documents = executeQuery($sql, [$numericMemberId]);
        
        http_response_code(200);
        echo json_encode([
            'count' => count($documents),
            'documents' => $documents
        ]);
    } catch (Exception $e) {
        error_log("Error fetching member documents: " . $e->getMessage());
        http_response_code(500);
        echo json_encode(['error' => 'Error fetching member documents: ' . $e->getMessage()]);
    }
}

function getDocumentsByCategory($category) {
    $sql = "SELECT d.id, d.title, d.description, d.file_path, d.file_type, d.file_size,
                   d.category, d.uploaded_by, d.related_member_id, d.related_meeting_id,
                   d.is_public, d.status, d.created_at, d.updated_at,
                   m.first_name, m.last_name, m.member_id as member_ref,
                   rm.first_name as related_member_first_name, rm.last_name as related_member_last_name,
                   rm.member_id as related_member_ref
            FROM documents d
            LEFT JOIN members m ON d.uploaded_by = m.id
            LEFT JOIN members rm ON d.related_member_id = rm.id
            WHERE d.category = ?
            ORDER BY d.created_at DESC";
    
    $documents = executeQuery($sql, [$category]);
    
    http_response_code(200);
    echo json_encode([
        'count' => count($documents),
        'documents' => $documents
    ]);
}

function getDocuments() {
    $category_filter = isset($_GET['category']) ? $_GET['category'] : null;
    $public_filter = isset($_GET['public']) ? (bool)$_GET['public'] : null;
    $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 50;
    $offset = isset($_GET['offset']) ? (int)$_GET['offset'] : 0;
    
    $sql = "SELECT d.id, d.title, d.description, d.file_path, d.file_type, d.file_size,
                   d.category, d.uploaded_by, d.related_member_id, d.related_meeting_id,
                   d.is_public, d.status, d.created_at, d.updated_at,
                   m.first_name, m.last_name, m.member_id as member_ref,
                   rm.first_name as related_member_first_name, rm.last_name as related_member_last_name,
                   rm.member_id as related_member_ref
            FROM documents d
            LEFT JOIN members m ON d.uploaded_by = m.id
            LEFT JOIN members rm ON d.related_member_id = rm.id";
    
    $params = [];
    $conditions = [];
    
    if ($category_filter) {
        $conditions[] = "d.category = ?";
        $params[] = $category_filter;
    }
    
    if ($public_filter !== null) {
        $conditions[] = "d.is_public = ?";
        $params[] = $public_filter ? 1 : 0;
    }
    
    if (!empty($conditions)) {
        $sql .= " WHERE " . implode(" AND ", $conditions);
    }
    
    $sql .= " ORDER BY d.created_at DESC LIMIT ? OFFSET ?";
    $params[] = $limit;
    $params[] = $offset;
    
    $documents = executeQuery($sql, $params);
    
    // Get total count
    $count_sql = "SELECT COUNT(*) as total FROM documents";
    $count_params = [];
    
    if (!empty($conditions)) {
        $count_sql .= " WHERE " . implode(" AND ", $conditions);
        $count_params = $params;
        // Remove the limit and offset parameters
        array_pop($count_params);
        array_pop($count_params);
    }
    
    $count_result = fetchSingleRow($count_sql, $count_params);
    $total = $count_result['total'];
    
    http_response_code(200);
    echo json_encode([
        'count' => count($documents),
        'total' => $total,
        'offset' => $offset,
        'limit' => $limit,
        'documents' => $documents
    ]);
}

function handlePostRequest() {
    // Check if this is a file upload request
    if (isset($_FILES['document'])) {
        handleFileUpload();
        return;
    }
    
    // Otherwise, handle JSON data
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid JSON data']);
        return;
    }
    
    // Validate required fields
    $required_fields = ['title', 'uploaded_by'];
    foreach ($required_fields as $field) {
        if (!isset($input[$field]) || empty($input[$field])) {
            http_response_code(400);
            echo json_encode(['error' => "Missing required field: $field"]);
            return;
        }
    }
    
    // Set default values
    if (!isset($input['is_public'])) {
        $input['is_public'] = false;
    }
    
    // Set default status if not provided
    if (!isset($input['status'])) {
        $input['status'] = 'pending';
    }
    
    // Insert document
    $sql = "INSERT INTO documents (title, description, file_path, file_type, file_size,
            category, uploaded_by, related_member_id, related_meeting_id, is_public, status) 
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
    
    $params = [
        $input['title'],
        $input['description'] ?? null,
        $input['file_path'] ?? null,
        $input['file_type'] ?? null,
        $input['file_size'] ?? null,
        $input['category'] ?? 'Other',
        $input['uploaded_by'],
        $input['related_member_id'] ?? null,
        $input['related_meeting_id'] ?? null,
        $input['is_public'] ? 1 : 0,
        $input['status']
    ];
    
    try {
        beginTransaction();
        
        executeNonQuery($sql, $params);
        $document_id = getDatabaseConnection()->lastInsertId();
        
        commitTransaction();
        
        // Create audit log for document creation
        createAuditLog(null, 'create', 'documents', $document_id, null, [
            'title' => $input['title'],
            'category' => $input['category'] ?? 'Other',
            'is_public' => $input['is_public'] ? 1 : 0,
            'status' => $input['status']
        ]);
        
        http_response_code(201);
        echo json_encode([
            'message' => 'Document recorded successfully',
            'document_id' => $document_id
        ]);
    } catch (Exception $e) {
        rollbackTransaction();
        throw $e;
    }
}

function handleFileUpload() {
    try {
        // Validate the uploaded file
        if (!isset($_FILES['document']) || $_FILES['document']['error'] !== UPLOAD_ERR_OK) {
            http_response_code(400);
            echo json_encode(['error' => 'File upload error']);
            return;
        }
        
        $file = $_FILES['document'];
        
        // Validate file type
        $allowed_types = ['application/pdf', 'image/jpeg', 'image/png', 'application/msword', 
                         'application/vnd.openxmlformats-officedocument.wordprocessingml.document'];
        if (!in_array($file['type'], $allowed_types)) {
            http_response_code(400);
            echo json_encode(['error' => 'Invalid file type. Only PDF, JPG, PNG, DOC, and DOCX files are allowed.']);
            return;
        }
        
        // Validate file size (10MB limit)
        if ($file['size'] > 10 * 1024 * 1024) {
            http_response_code(400);
            echo json_encode(['error' => 'File size exceeds 10MB limit.']);
            return;
        }
        
        // Get other form data
        $title = isset($_POST['title']) ? $_POST['title'] : $file['name'];
        $description = isset($_POST['description']) ? $_POST['description'] : null;
        $category = isset($_POST['category']) ? $_POST['category'] : 'Other';
        $uploaded_by = isset($_POST['uploaded_by']) ? (int)$_POST['uploaded_by'] : null;
        $related_member_id = isset($_POST['related_member_id']) ? (int)$_POST['related_member_id'] : null;
        
        // Validate required fields
        if (!$uploaded_by) {
            http_response_code(400);
            echo json_encode(['error' => 'Uploaded by field is required']);
            return;
        }
        
        // Create category-specific upload directory
        $base_upload_dir = '../documents';
        $category_dir = strtolower($category);
        
        // Map specific categories to appropriate directories
        switch (strtolower($category)) {
            case 'kyc':
                $upload_dir = $base_upload_dir . '/kyc';
                break;
            case 'agreement':
                $upload_dir = $base_upload_dir . '/loan_agreements';
                break;
            case 'receipt':
                $upload_dir = $base_upload_dir . '/receipts';
                break;
            case 'report':
                $upload_dir = $base_upload_dir . '/reports';
                break;
            case 'meeting':
                $upload_dir = $base_upload_dir . '/meetings';
                break;
            default:
                $upload_dir = $base_upload_dir . '/other';
                break;
        }
        
        // Create upload directory if it doesn't exist
        if (!file_exists($upload_dir)) {
            mkdir($upload_dir, 0755, true);
        }
        
        // Generate unique filename
        $file_extension = pathinfo($file['name'], PATHINFO_EXTENSION);
        $unique_filename = uniqid() . '_' . time() . '.' . $file_extension;
        $file_path = $upload_dir . '/' . $unique_filename;
        
        // Move uploaded file
        if (!move_uploaded_file($file['tmp_name'], $file_path)) {
            http_response_code(500);
            echo json_encode(['error' => 'Failed to move uploaded file']);
            return;
        }
        
        // Insert document record with default status 'available'
        $sql = "INSERT INTO documents (title, description, file_path, file_type, file_size,
                category, uploaded_by, related_member_id, is_public, status) 
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
        
        $params = [
            $title,
            $description,
            $file_path,
            $file['type'],
            $file['size'],
            $category,
            $uploaded_by,
            $related_member_id,
            0, // is_public
            'available' // status
        ];
        
        beginTransaction();
        
        executeNonQuery($sql, $params);
        $document_id = getDatabaseConnection()->lastInsertId();
        
        commitTransaction();
        
        // Create audit log for document creation
        createAuditLog(null, 'create', 'documents', $document_id, null, [
            'title' => $title,
            'category' => $category,
            'file_size' => $file['size']
        ]);
        
        http_response_code(201);
        echo json_encode([
            'message' => 'Document uploaded successfully',
            'document_id' => $document_id,
            'file_path' => $file_path
        ]);
    } catch (Exception $e) {
        rollbackTransaction();
        error_log("File upload error: " . $e->getMessage());
        http_response_code(500);
        echo json_encode(['error' => 'Error uploading document: ' . $e->getMessage()]);
    }
}

function handlePutRequest() {
    if (!isset($_GET['id'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Document ID is required']);
        return;
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid JSON data']);
        return;
    }
    
    // Update document
    $sql = "UPDATE documents SET 
            title = ?, 
            description = ?, 
            file_path = ?, 
            file_type = ?, 
            file_size = ?, 
            category = ?, 
            uploaded_by = ?, 
            related_member_id = ?, 
            related_meeting_id = ?, 
            is_public = ?
            WHERE id = ?";
    
    $params = [
        $input['title'] ?? null,
        $input['description'] ?? null,
        $input['file_path'] ?? null,
        $input['file_type'] ?? null,
        $input['file_size'] ?? null,
        $input['category'] ?? null,
        $input['uploaded_by'] ?? null,
        $input['related_member_id'] ?? null,
        $input['related_meeting_id'] ?? null,
        isset($input['is_public']) ? ($input['is_public'] ? 1 : 0) : null,
        $_GET['id']
    ];
    
    // Remove null values
    $fields = ['title', 'description', 'file_path', 'file_type', 'file_size', 'category', 
               'uploaded_by', 'related_member_id', 'related_meeting_id', 'is_public', 'id'];
    $filtered_params = [];
    $set_parts = [];
    
    for ($i = 0; $i < count($fields) - 1; $i++) {
        if ($params[$i] !== null) {
            $set_parts[] = $fields[$i] . " = ?";
            $filtered_params[] = $params[$i];
        }
    }
    
    if (empty($set_parts)) {
        http_response_code(400);
        echo json_encode(['error' => 'No fields to update']);
        return;
    }
    
    $sql = "UPDATE documents SET " . implode(", ", $set_parts) . " WHERE id = ?";
    $filtered_params[] = $_GET['id'];
    
    try {
        beginTransaction();
        
        $rows_affected = executeNonQuery($sql, $filtered_params);
        
        if ($rows_affected === 0) {
            http_response_code(404);
            echo json_encode(['error' => 'Document not found']);
            return;
        }
        
        commitTransaction();
        
        // Create audit log for document update
        createAuditLog(null, 'update', 'documents', $_GET['id'], null, $input);
        
        http_response_code(200);
        echo json_encode([
            'message' => 'Document updated successfully'
        ]);
    } catch (Exception $e) {
        rollbackTransaction();
        throw $e;
    }
}

function handleDeleteRequest() {
    if (!isset($_GET['id'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Document ID is required']);
        return;
    }
    
    try {
        // First, get the document to check if it can be deleted
        error_log("Attempting to delete document ID: " . $_GET['id']);
        
        // Let's add some debugging to see what database we're connected to
        $pdo = getDatabaseConnection();
        $db_name_stmt = $pdo->prepare("SELECT DATABASE() as db_name");
        $db_name_stmt->execute();
        $db_result = $db_name_stmt->fetch();
        error_log("Connected to database: " . $db_result['db_name']);
        
        // Check if the members table has the role column
        $column_check_stmt = $pdo->prepare("SHOW COLUMNS FROM members LIKE 'role'");
        $column_check_stmt->execute();
        $column_exists = $column_check_stmt->fetch();
        error_log("Role column exists: " . ($column_exists ? "YES" : "NO"));
        
        $documentSql = "SELECT d.id, d.title, d.file_path, d.uploaded_by, d.related_member_id, d.category, m.role as uploader_role
                        FROM documents d
                        LEFT JOIN members m ON d.uploaded_by = m.id
                        WHERE d.id = ?";
        
        error_log("Executing query: $documentSql with ID: " . $_GET['id']);
        
        $document = fetchSingleRow($documentSql, [$_GET['id']]);
        
        error_log("Query result: " . print_r($document, true));
        
        if (!$document) {
            http_response_code(404);
            echo json_encode(['error' => 'Document not found']);
            return;
        }
        
        // Check if the document was uploaded by an admin (role = 'admin' or 'Admin')
        // In this case, members cannot delete admin-uploaded documents
        if (isset($document['uploader_role']) && 
            (strtolower($document['uploader_role']) === 'admin' || 
             strpos(strtolower($document['uploader_role']), 'admin') !== false)) {
            http_response_code(403);
            echo json_encode(['error' => 'You cannot delete documents uploaded by administrators']);
            return;
        }
        
        // Check if the document belongs to a meeting (admin-generated)
        // This is a simplified check - in a real implementation, you might have a more specific way to identify admin documents
        $adminDocumentCategories = ['Agreement', 'Meeting', 'Report'];
        if (in_array($document['category'], $adminDocumentCategories)) {
            http_response_code(403);
            echo json_encode(['error' => 'You cannot delete administrator-generated documents']);
            return;
        }
        
        // If we reach here, the document can be deleted
        $sql = "DELETE FROM documents WHERE id = ?";
        
        beginTransaction();
        
        $rows_affected = executeNonQuery($sql, [$_GET['id']]);
        
        if ($rows_affected === 0) {
            http_response_code(404);
            echo json_encode(['error' => 'Document not found']);
            return;
        }
        
        commitTransaction();
        
        // Create audit log for document deletion
        createAuditLog(null, 'delete', 'documents', $_GET['id'], null, null);
        
        // Delete the file from the filesystem if it exists
        if (!empty($document['file_path']) && file_exists($document['file_path'])) {
            unlink($document['file_path']);
        }
        
        http_response_code(200);
        echo json_encode([
            'message' => 'Document deleted successfully'
        ]);
    } catch (Exception $e) {
        rollbackTransaction();
        error_log("Error deleting document: " . $e->getMessage());
        error_log("Stack trace: " . $e->getTraceAsString());
        http_response_code(500);
        echo json_encode(['error' => 'Error deleting document: ' . $e->getMessage()]);
    }
}
?>