<?php
// MFT GROUP Dashboard API
// This API provides summary data for the admin and member dashboards

// Enable CORS for frontend access - more permissive headers
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Credentials: true");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS");
header("Access-Control-Allow-Headers: Origin, X-Requested-With, Content-Type, Accept, Authorization");
header("Content-Type: application/json; charset=UTF-8");

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Include database configuration with proper path resolution
$rootPath = dirname(__DIR__);
require_once $rootPath . '/database/config.php';

// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('log_errors', 1);

// Get request method
$method = $_SERVER['REQUEST_METHOD'];

if ($method !== 'GET') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit;
}

// Get dashboard type from query parameter
$dashboard_type = isset($_GET['type']) ? $_GET['type'] : 'admin';
$action = isset($_GET['action']) ? $_GET['action'] : '';

// Add debugging information
error_log("Dashboard API called with type: $dashboard_type, action: $action");

try {
    if ($action === 'contributions_summary') {
        getContributionsSummary();
    } else if ($action === 'getAllAdminData') {
        getAllAdminData();
    } else if ($action === 'getAdminStats') {
        getAdminStats();
    } else if ($action === 'getRecentActivities') {
        getRecentActivities();
    } else if ($action === 'getFinancialOverview') {
        getFinancialOverview();
    } else if ($action === 'getAllData' && $dashboard_type === 'member' && isset($_GET['member_id'])) {
        getMemberDashboardData($_GET['member_id']);
    } else if ($dashboard_type === 'admin') {
        getAdminDashboardData();
    } else if ($dashboard_type === 'member' && isset($_GET['member_id'])) {
        getMemberDashboardData($_GET['member_id']);
    } else {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid dashboard type or missing member_id for member dashboard']);
    }
} catch (Exception $e) {
    error_log("Error in dashboard API: " . $e->getMessage() . " in " . $e->getFile() . " on line " . $e->getLine());
    http_response_code(500);
    echo json_encode(['error' => 'Internal server error: ' . $e->getMessage()]);
}

function getContributionsSummary() {
    $data = [];
    
    try {
        // Get total contributions
        $total_contributions_sql = "SELECT COALESCE(SUM(amount), 0) as total_contributions FROM contributions WHERE status = 'Confirmed'";
        $total_result = fetchSingleRow($total_contributions_sql);
        $data['total_contributions'] = (float)($total_result['total_contributions'] ?? 0);
        
        // Get active members
        $active_members_sql = "SELECT COUNT(*) as active_members FROM members WHERE status = 'Active'";
        $active_result = fetchSingleRow($active_members_sql);
        $data['active_members'] = (int)($active_result['active_members'] ?? 0);
        
        // Get total members
        $total_members_sql = "SELECT COUNT(*) as total_members FROM members WHERE status != 'Pending'";
        $total_result = fetchSingleRow($total_members_sql);
        $data['total_members'] = (int)($total_result['total_members'] ?? 0);
        
        // Calculate collection rate (simplified calculation)
        $data['collection_rate'] = $data['total_members'] > 0 ? round(($data['active_members'] / $data['total_members']) * 100, 2) : 0;
        
        // Get pending amount
        $pending_amount_sql = "SELECT COALESCE(SUM(amount), 0) as pending_amount FROM contributions WHERE status = 'Pending'";
        $pending_result = fetchSingleRow($pending_amount_sql);
        $data['pending_amount'] = (float)($pending_result['pending_amount'] ?? 0);
        
        http_response_code(200);
        echo json_encode($data);
    } catch (Exception $e) {
        error_log("Error getting contributions summary: " . $e->getMessage());
        http_response_code(500);
        echo json_encode(['error' => 'Internal server error: ' . $e->getMessage()]);
    }
}

function getAllAdminData() {
    $data = [];
    
    try {
        // Get admin stats
        $stats = getAdminStatsData();
        $data['stats'] = $stats;
        
        // Get recent activities
        $activities = getRecentActivitiesData();
        $data['activities'] = $activities;
        
        // Get financial overview
        $financial = getFinancialOverviewData();
        $data['financial'] = $financial;
        
        http_response_code(200);
        echo json_encode($data);
    } catch (Exception $e) {
        error_log("Error getting all admin data: " . $e->getMessage());
        http_response_code(500);
        echo json_encode(['error' => 'Internal server error: ' . $e->getMessage()]);
    }
}

function getAdminStatsData() {
    $data = [];
    
    try {
        // Get member statistics
        $member_sql = "SELECT COUNT(*) as total, 
                              SUM(CASE WHEN status = 'Active' THEN 1 ELSE 0 END) as active,
                              SUM(CASE WHEN status = 'Pending' THEN 1 ELSE 0 END) as pending
                       FROM members";
        $member_result = fetchSingleRow($member_sql);
        // Ensure numeric values
        $data['members'] = [
            'total' => (int)($member_result['total'] ?? 0),
            'active' => (int)($member_result['active'] ?? 0),
            'pending' => (int)($member_result['pending'] ?? 0)
        ];
        
        // Get contribution statistics
        $contributions_sql = "SELECT COALESCE(SUM(amount), 0) as total_balance,
                                     COALESCE(SUM(CASE WHEN YEAR(contribution_date) = YEAR(CURDATE()) AND MONTH(contribution_date) = MONTH(CURDATE()) THEN amount ELSE 0 END), 0) as this_month
                              FROM contributions 
                              WHERE status = 'Confirmed'";
        $contributions_result = fetchSingleRow($contributions_sql);
        // Ensure numeric values
        $data['contributions'] = [
            'total_balance' => (float)($contributions_result['total_balance'] ?? 0),
            'this_month' => (float)($contributions_result['this_month'] ?? 0)
        ];
        
        // Get loan statistics
        $loans_sql = "SELECT COALESCE(SUM(CASE WHEN status = 'Active' THEN principal_amount ELSE 0 END), 0) as outstanding_balance,
                             SUM(CASE WHEN status = 'Active' THEN 1 ELSE 0 END) as active_loans,
                             SUM(CASE WHEN status = 'Pending' THEN 1 ELSE 0 END) as pending_applications
                      FROM loans 
                      WHERE status IN ('Active', 'Pending')";
        $loans_result = fetchSingleRow($loans_sql);
        // Ensure numeric values
        $data['loans'] = [
            'outstanding_balance' => (float)($loans_result['outstanding_balance'] ?? 0),
            'active_loans' => (int)($loans_result['active_loans'] ?? 0),
            'pending_applications' => (int)($loans_result['pending_applications'] ?? 0)
        ];
        
        // Get project statistics
        $projects_sql = "SELECT COUNT(*) as active,
                                SUM(CASE WHEN status = 'Completed' THEN 1 ELSE 0 END) as completed
                         FROM projects 
                         WHERE status IN ('Active', 'Completed')";
        $projects_result = fetchSingleRow($projects_sql);
        // Ensure numeric values
        $data['projects'] = [
            'active' => (int)($projects_result['active'] ?? 0),
            'completed' => (int)($projects_result['completed'] ?? 0)
        ];
        
        // Get pending fines
        $fines_sql = "SELECT COALESCE(SUM(amount), 0) as total_pending_fines,
                             COUNT(*) as pending_fines_count
                      FROM fines 
                      WHERE status IN ('Pending', 'Unpaid')";
        $fines_result = fetchSingleRow($fines_sql);
        // Ensure numeric values
        $data['fines'] = [
            'total_pending_fines' => (float)($fines_result['total_pending_fines'] ?? 0),
            'pending_fines_count' => (int)($fines_result['pending_fines_count'] ?? 0)
        ];
        
        // Get paid fines (for Bank Balance calculation)
        $paid_fines_sql = "SELECT COALESCE(SUM(amount), 0) as total_paid_fines
                          FROM fines 
                          WHERE status = 'Paid'";
        $paid_fines_result = fetchSingleRow($paid_fines_sql);
        // Ensure numeric values
        $data['paid_fines'] = [
            'total_paid_fines' => (float)($paid_fines_result['total_paid_fines'] ?? 0)
        ];
        
        // Calculate financial summary
        $financial_summary_sql = "SELECT 
                                         (SELECT COALESCE(SUM(amount), 0) FROM contributions WHERE status = 'Confirmed') as total_contributions,
                                         (SELECT COALESCE(SUM(amount), 0) FROM fines WHERE status = 'Paid') as total_paid_fines,
                                         (SELECT COALESCE(SUM(principal_amount), 0) FROM loans WHERE status = 'Active') as total_loans,
                                         (SELECT COALESCE(SUM(balance), 0) FROM accounts WHERE account_type_id = (SELECT id FROM account_types WHERE name = 'Expense')) as total_expenses";
        $financial_data = fetchSingleRow($financial_summary_sql);
        
        $data['financial_summary'] = [
            'total_contributions' => (float)($financial_data['total_contributions'] ?? 0),
            'total_paid_fines' => (float)($financial_data['total_paid_fines'] ?? 0),
            'total_loans' => (float)($financial_data['total_loans'] ?? 0),
            'total_expenses' => (float)($financial_data['total_expenses'] ?? 0),
            'bank_balance' => (float)(($financial_data['total_contributions'] ?? 0) + ($financial_data['total_paid_fines'] ?? 0)),
            'total_withdrawals' => (float)($financial_data['total_loans'] ?? 0),
            'net_balance' => (float)(($financial_data['total_contributions'] ?? 0) + ($financial_data['total_paid_fines'] ?? 0) - ($financial_data['total_loans'] ?? 0) - ($financial_data['total_expenses'] ?? 0))
        ];
        
        return $data;
    } catch (Exception $e) {
        error_log("Error getting admin stats data: " . $e->getMessage());
        throw $e;
    }
}

function getRecentActivitiesData() {
    try {
        // Get recent activities (last 10 activities)
        $activities_sql = "
            SELECT 'contribution' as type, 
                   CONCAT(m.first_name, ' ', m.last_name, ' made a contribution of KSH ', FORMAT(c.amount, 0)) as description,
                   c.created_at as timestamp
            FROM contributions c
            JOIN members m ON c.member_id = m.id
            WHERE c.status = 'Confirmed'
            UNION ALL
            SELECT 'loan' as type,
                   CONCAT(m.first_name, ' ', m.last_name, ' applied for a loan of KSH ', FORMAT(l.principal_amount, 0)) as description,
                   l.created_at as timestamp
            FROM loans l
            JOIN members m ON l.member_id = m.id
            WHERE l.status = 'Pending'
            UNION ALL
            SELECT 'member' as type,
                   CONCAT('New member ', m.first_name, ' ', m.last_name, ' registered (', m.member_id, ')') as description,
                   m.created_at as timestamp
            FROM members m
            WHERE m.status != 'Pending'
            UNION ALL
            SELECT 'project' as type,
                   CONCAT('Project \"', p.name, '\" status updated to ', p.status) as description,
                   p.updated_at as timestamp
            FROM projects p
            WHERE p.status IN ('Active', 'Completed')
            ORDER BY timestamp DESC
            LIMIT 10";
        
        return executeQuery($activities_sql);
    } catch (Exception $e) {
        error_log("Error getting recent activities data: " . $e->getMessage());
        throw $e;
    }
}

function getFinancialOverviewData() {
    $data = [];
    
    try {
        // Get account balances
        $accounts_sql = "SELECT a.name, a.balance, 
                                CASE 
                                    WHEN at.name = 'Savings' THEN 'savings'
                                    WHEN at.name = 'Loan' THEN 'loan'
                                    WHEN at.name = 'Welfare' THEN 'welfare'
                                    WHEN at.name = 'Project' THEN 'project'
                                    WHEN at.name = 'Dividend' THEN 'dividend'
                                    WHEN at.name = 'Expense' THEN 'expense'
                                    ELSE 'other'
                                END as type
                         FROM accounts a
                         JOIN account_types at ON a.account_type_id = at.id
                         WHERE a.is_active = 1
                         ORDER BY a.id";
        $data['accounts'] = executeQuery($accounts_sql);
        
        return $data;
    } catch (Exception $e) {
        error_log("Error getting financial overview data: " . $e->getMessage());
        throw $e;
    }
}

function getAdminStats() {
    $data = [];
    
    try {
        // Get member statistics
        $member_sql = "SELECT COUNT(*) as total, 
                              SUM(CASE WHEN status = 'Active' THEN 1 ELSE 0 END) as active,
                              SUM(CASE WHEN status = 'Pending' THEN 1 ELSE 0 END) as pending
                       FROM members";
        $member_result = fetchSingleRow($member_sql);
        // Ensure numeric values
        $data['members'] = [
            'total' => (int)($member_result['total'] ?? 0),
            'active' => (int)($member_result['active'] ?? 0),
            'pending' => (int)($member_result['pending'] ?? 0)
        ];
        
        // Get contribution statistics
        $contributions_sql = "SELECT COALESCE(SUM(amount), 0) as total_balance,
                                     COALESCE(SUM(CASE WHEN YEAR(contribution_date) = YEAR(CURDATE()) AND MONTH(contribution_date) = MONTH(CURDATE()) THEN amount ELSE 0 END), 0) as this_month
                              FROM contributions 
                              WHERE status = 'Confirmed'";
        $contributions_result = fetchSingleRow($contributions_sql);
        // Ensure numeric values
        $data['contributions'] = [
            'total_balance' => (float)($contributions_result['total_balance'] ?? 0),
            'this_month' => (float)($contributions_result['this_month'] ?? 0)
        ];
        
        // Get loan statistics
        $loans_sql = "SELECT COALESCE(SUM(CASE WHEN status = 'Active' THEN principal_amount ELSE 0 END), 0) as outstanding_balance,
                             SUM(CASE WHEN status = 'Active' THEN 1 ELSE 0 END) as active_loans,
                             SUM(CASE WHEN status = 'Pending' THEN 1 ELSE 0 END) as pending_applications
                      FROM loans 
                      WHERE status IN ('Active', 'Pending')";
        $loans_result = fetchSingleRow($loans_sql);
        // Ensure numeric values
        $data['loans'] = [
            'outstanding_balance' => (float)($loans_result['outstanding_balance'] ?? 0),
            'active_loans' => (int)($loans_result['active_loans'] ?? 0),
            'pending_applications' => (int)($loans_result['pending_applications'] ?? 0)
        ];
        
        // Get project statistics
        $projects_sql = "SELECT COUNT(*) as active,
                                SUM(CASE WHEN status = 'Completed' THEN 1 ELSE 0 END) as completed
                         FROM projects 
                         WHERE status IN ('Active', 'Completed')";
        $projects_result = fetchSingleRow($projects_sql);
        // Ensure numeric values
        $data['projects'] = [
            'active' => (int)($projects_result['active'] ?? 0),
            'completed' => (int)($projects_result['completed'] ?? 0)
        ];
        
        // Get pending fines
        $fines_sql = "SELECT COALESCE(SUM(amount), 0) as total_pending_fines,
                             COUNT(*) as pending_fines_count
                      FROM fines 
                      WHERE status IN ('Pending', 'Unpaid')";
        $fines_result = fetchSingleRow($fines_sql);
        // Ensure numeric values
        $data['fines'] = [
            'total_pending_fines' => (float)($fines_result['total_pending_fines'] ?? 0),
            'pending_fines_count' => (int)($fines_result['pending_fines_count'] ?? 0)
        ];
        
        // Get paid fines (for Bank Balance calculation)
        $paid_fines_sql = "SELECT COALESCE(SUM(amount), 0) as total_paid_fines
                          FROM fines 
                          WHERE status = 'Paid'";
        $paid_fines_result = fetchSingleRow($paid_fines_sql);
        // Ensure numeric values
        $data['paid_fines'] = [
            'total_paid_fines' => (float)($paid_fines_result['total_paid_fines'] ?? 0)
        ];
        
        // Calculate financial summary
        $financial_summary_sql = "SELECT 
                                         (SELECT COALESCE(SUM(amount), 0) FROM contributions WHERE status = 'Confirmed') as total_contributions,
                                         (SELECT COALESCE(SUM(amount), 0) FROM fines WHERE status = 'Paid') as total_paid_fines,
                                         (SELECT COALESCE(SUM(principal_amount), 0) FROM loans WHERE status = 'Active') as total_loans,
                                         (SELECT COALESCE(SUM(balance), 0) FROM accounts WHERE account_type_id = (SELECT id FROM account_types WHERE name = 'Expense')) as total_expenses";
        $financial_data = fetchSingleRow($financial_summary_sql);
        
        $data['financial_summary'] = [
            'total_contributions' => (float)($financial_data['total_contributions'] ?? 0),
            'total_paid_fines' => (float)($financial_data['total_paid_fines'] ?? 0),
            'total_loans' => (float)($financial_data['total_loans'] ?? 0),
            'total_expenses' => (float)($financial_data['total_expenses'] ?? 0),
            'bank_balance' => (float)(($financial_data['total_contributions'] ?? 0) + ($financial_data['total_paid_fines'] ?? 0)),
            'total_withdrawals' => (float)($financial_data['total_loans'] ?? 0),
            'net_balance' => (float)(($financial_data['total_contributions'] ?? 0) + ($financial_data['total_paid_fines'] ?? 0) - ($financial_data['total_loans'] ?? 0) - ($financial_data['total_expenses'] ?? 0))
        ];
        
        http_response_code(200);
        echo json_encode($data);
    } catch (Exception $e) {
        error_log("Error getting admin stats: " . $e->getMessage());
        throw $e;
    }
}

function getRecentActivities() {
    $data = [];
    
    try {
        // Get recent activities (last 10 activities)
        $activities_sql = "
            SELECT 'contribution' as type, 
                   CONCAT(m.first_name, ' ', m.last_name, ' made a contribution of KSH ', FORMAT(c.amount, 0)) as description,
                   c.created_at as timestamp
            FROM contributions c
            JOIN members m ON c.member_id = m.id
            WHERE c.status = 'Confirmed'
            UNION ALL
            SELECT 'loan' as type,
                   CONCAT(m.first_name, ' ', m.last_name, ' applied for a loan of KSH ', FORMAT(l.principal_amount, 0)) as description,
                   l.created_at as timestamp
            FROM loans l
            JOIN members m ON l.member_id = m.id
            WHERE l.status = 'Pending'
            UNION ALL
            SELECT 'member' as type,
                   CONCAT('New member ', m.first_name, ' ', m.last_name, ' registered (', m.member_id, ')') as description,
                   m.created_at as timestamp
            FROM members m
            WHERE m.status != 'Pending'
            UNION ALL
            SELECT 'project' as type,
                   CONCAT('Project \"', p.name, '\" status updated to ', p.status) as description,
                   p.updated_at as timestamp
            FROM projects p
            WHERE p.status IN ('Active', 'Completed')
            ORDER BY timestamp DESC
            LIMIT 10";
        
        $data = executeQuery($activities_sql);
        
        http_response_code(200);
        echo json_encode($data);
    } catch (Exception $e) {
        error_log("Error getting recent activities: " . $e->getMessage());
        throw $e;
    }
}

function getFinancialOverview() {
    $data = [];
    
    try {
        // Get account balances
        $accounts_sql = "SELECT a.name, a.balance, 
                                CASE 
                                    WHEN at.name = 'Savings' THEN 'savings'
                                    WHEN at.name = 'Loan' THEN 'loan'
                                    WHEN at.name = 'Welfare' THEN 'welfare'
                                    WHEN at.name = 'Project' THEN 'project'
                                    WHEN at.name = 'Dividend' THEN 'dividend'
                                    WHEN at.name = 'Expense' THEN 'expense'
                                    ELSE 'other'
                                END as type
                         FROM accounts a
                         JOIN account_types at ON a.account_type_id = at.id
                         WHERE a.is_active = 1
                         ORDER BY a.id";
        $data['accounts'] = executeQuery($accounts_sql);
        
        http_response_code(200);
        echo json_encode($data);
    } catch (Exception $e) {
        error_log("Error getting financial overview: " . $e->getMessage());
        throw $e;
    }
}

function getAdminDashboardData() {
    $data = [];
    
    // Get total members
    $member_sql = "SELECT COUNT(*) as total, 
                          SUM(CASE WHEN status = 'Active' THEN 1 ELSE 0 END) as active,
                          SUM(CASE WHEN status = 'Pending' THEN 1 ELSE 0 END) as pending
                   FROM members";
    $data['members'] = fetchSingleRow($member_sql);
    
    // Get total contributions this month
    $contribution_sql = "SELECT COALESCE(SUM(amount), 0) as total,
                                COUNT(*) as count
                         FROM contributions 
                         WHERE YEAR(contribution_date) = YEAR(CURDATE()) 
                         AND MONTH(contribution_date) = MONTH(CURDATE())";
    $data['contributions_this_month'] = fetchSingleRow($contribution_sql);
    
    // Get pending loans
    $pending_loans_sql = "SELECT COUNT(*) as count,
                                 COALESCE(SUM(principal_amount), 0) as total_amount
                          FROM loans 
                          WHERE status = 'Pending'";
    $data['pending_loans'] = fetchSingleRow($pending_loans_sql);
    
    // Get active loans
    $active_loans_sql = "SELECT COUNT(*) as count,
                                COALESCE(SUM(principal_amount), 0) as total_amount,
                                COALESCE(SUM(principal_amount), 0) as total_with_interest
                         FROM loans 
                         WHERE status = 'Active'";
    $data['active_loans'] = fetchSingleRow($active_loans_sql);
    
    // Get account balances
    $accounts_sql = "SELECT a.name, a.balance, a.account_type_id
                     FROM accounts a
                     WHERE a.is_active = 1
                     ORDER BY a.id";
    $data['accounts'] = executeQuery($accounts_sql);
    
    // Get recent contributions
    $recent_contributions_sql = "SELECT c.amount, c.contribution_date, m.first_name, m.last_name
                                 FROM contributions c
                                 JOIN members m ON c.member_id = m.id
                                 ORDER BY c.contribution_date DESC
                                 LIMIT 5";
    $data['recent_contributions'] = executeQuery($recent_contributions_sql);
    
    // Get recent loan applications
    $recent_loans_sql = "SELECT l.reference, l.purpose as loan_type, l.principal_amount, l.status, 
                                m.first_name, m.last_name, l.application_date
                         FROM loans l
                         JOIN members m ON l.member_id = m.id
                         ORDER BY l.application_date DESC
                         LIMIT 5";
    $data['recent_loans'] = executeQuery($recent_loans_sql);
    
    http_response_code(200);
    echo json_encode($data);
}

function getMemberRecentActivities($member_id) {
    $data = [];
    
    try {
        // First get the member's numeric ID
        $member_check_sql = "SELECT id FROM members WHERE member_id = ?";
        $member_result = fetchSingleRow($member_check_sql, [$member_id]);
        
        if (!$member_result) {
            throw new Exception('Member not found');
        }
        
        $member_numeric_id = $member_result['id'];
        
        // Get recent activities for this specific member (last 10 activities)
        $activities_sql = "
            SELECT 'contribution' as type, 
                   CONCAT('You made a contribution of KSH ', FORMAT(c.amount, 0)) as description,
                   c.created_at as timestamp
            FROM contributions c
            WHERE c.member_id = ? AND c.status = 'Confirmed'
            UNION ALL
            SELECT 'loan' as type,
                   CASE 
                       WHEN l.status = 'Pending' THEN CONCAT('You applied for a loan of KSH ', FORMAT(l.principal_amount, 0))
                       WHEN l.status = 'Approved' THEN CONCAT('Your loan of KSH ', FORMAT(l.principal_amount, 0), ' was approved')
                       WHEN l.status = 'Rejected' THEN CONCAT('Your loan application for KSH ', FORMAT(l.principal_amount, 0), ' was rejected')
                       WHEN l.status = 'Active' THEN CONCAT('Your loan of KSH ', FORMAT(l.principal_amount, 0), ' is now active')
                       WHEN l.status = 'Completed' THEN CONCAT('You completed your loan of KSH ', FORMAT(l.principal_amount, 0))
                       ELSE CONCAT('Loan status updated: ', l.status)
                   END as description,
                   l.created_at as timestamp
            FROM loans l
            WHERE l.member_id = ?
            UNION ALL
            SELECT 'fine' as type,
                   CASE 
                       WHEN f.status = 'Pending' THEN CONCAT('You have a new fine of KSH ', FORMAT(f.amount, 0))
                       WHEN f.status = 'Paid' THEN CONCAT('You paid a fine of KSH ', FORMAT(f.amount, 0))
                       WHEN f.status = 'Unpaid' THEN CONCAT('You have an unpaid fine of KSH ', FORMAT(f.amount, 0))
                       ELSE CONCAT('Fine status updated: ', f.status)
                   END as description,
                   f.created_at as timestamp
            FROM fines f
            WHERE f.member_id = ?
            UNION ALL
            SELECT 'project' as type,
                   CASE 
                       WHEN mp.status = 'Active' THEN CONCAT('You joined the \"', p.name, '\" project')
                       WHEN mp.status = 'Completed' THEN CONCAT('You completed the \"', p.name, '\" project')
                       ELSE CONCAT('Project \"', p.name, '\" status updated to ', mp.status)
                   END as description,
                   mp.join_date as timestamp
            FROM projects p
            JOIN member_projects mp ON p.id = mp.project_id
            WHERE mp.member_id = ?
            ORDER BY timestamp DESC
            LIMIT 10";
        
        $data = executeQuery($activities_sql, [$member_numeric_id, $member_numeric_id, $member_numeric_id, $member_numeric_id]);
        
        return $data;
    } catch (Exception $e) {
        error_log("Error getting member recent activities: " . $e->getMessage());
        return [];
    }
}

// Add the recent activities data to the member dashboard data
function getMemberDashboardData($member_id) {
    $data = [];
    
    try {
        // Get member info including last_login and last_login_ip
        // First, we need to get the member's numeric ID from the member_id
        $member_check_sql = "SELECT id, first_name, last_name, member_id, status, registration_date, last_login, last_login_ip
                             FROM members 
                             WHERE member_id = ?";
        $member_info = fetchSingleRow($member_check_sql, [$member_id]);
        
        if (!$member_info) {
            http_response_code(404);
            echo json_encode(['error' => 'Member not found']);
            return;
        }
        
        // Use the numeric ID for subsequent queries
        $member_numeric_id = $member_info['id'];
        
        // Get member contributions balance
        $contributions_sql = "SELECT COALESCE(SUM(amount), 0) as total_balance
                              FROM contributions 
                              WHERE member_id = ? AND status = 'Confirmed'";
        $data['contributions'] = fetchSingleRow($contributions_sql, [$member_numeric_id]);
        
        // Get member outstanding loans
        $loans_sql = "SELECT COALESCE(SUM(principal_amount), 0) as outstanding_balance,
                             COUNT(*) as active_loans
                      FROM loans 
                      WHERE member_id = ? AND status = 'Active'";
        $data['loans'] = fetchSingleRow($loans_sql, [$member_numeric_id]);
        
        // Get member fines (not dividends as in the previous code)
        $fines_sql = "SELECT COALESCE(SUM(amount), 0) as total_fines
                      FROM fines 
                      WHERE member_id = ? AND status IN ('Pending', 'Unpaid')";
        $data['fines'] = fetchSingleRow($fines_sql, [$member_numeric_id]);
        
        // Get member projects
        $projects_sql = "SELECT p.id, p.name, p.description, p.location, p.created_at as start_date, p.completion_date, p.completion_date,
                                p.status as project_status, mp.status as member_project_status, 
                                mp.amount_contributed as contribution_amount, mp.join_date as joined_date
                         FROM projects p
                         JOIN member_projects mp ON p.id = mp.project_id
                         WHERE mp.member_id = ?
                         ORDER BY p.created_at DESC";
        $data['projects'] = executeQuery($projects_sql, [$member_numeric_id]);
        
        // Get recent contributions
        $recent_contributions_sql = "SELECT amount, contribution_date, status
                                     FROM contributions 
                                     WHERE member_id = ?
                                     ORDER BY contribution_date DESC
                                     LIMIT 5";
        $data['recent_contributions'] = executeQuery($recent_contributions_sql, [$member_numeric_id]);
        
        // Get recent loans
        $recent_loans_sql = "SELECT reference, purpose as loan_type, principal_amount, status, application_date
                             FROM loans 
                             WHERE member_id = ?
                             ORDER BY application_date DESC
                             LIMIT 5";
        $data['recent_loans'] = executeQuery($recent_loans_sql, [$member_numeric_id]);
        
        // Get recent activities
        $data['recent_activities'] = getMemberRecentActivities($member_id);
        
        // Include member info in the response
        $data['member_info'] = $member_info;
        
        http_response_code(200);
        echo json_encode($data);
    } catch (Exception $e) {
        error_log("Error in getMemberDashboardData: " . $e->getMessage());
        throw $e;
    }
}
?>