<?php
// Banking Settings API
// This script handles CRUD operations for banking settings

// Enable CORS for frontend access
header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");

// Handle preflight OPTIONS request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// Include database configuration
require_once __DIR__ . '/../database/config.php';

// Get request method
$method = $_SERVER['REQUEST_METHOD'];

try {
    switch ($method) {
        case 'GET':
            handleGetRequest();
            break;
        case 'POST':
            handlePostRequest();
            break;
        case 'PUT':
            handlePutRequest();
            break;
        case 'DELETE':
            handleDeleteRequest();
            break;
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            break;
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Internal server error: ' . $e->getMessage()]);
}

function handleGetRequest() {
    if (isset($_GET['id'])) {
        // Get specific banking setting by ID
        getBankingSetting($_GET['id']);
    } else {
        // Get all banking settings
        getBankingSettings();
    }
}

function getBankingSetting($id) {
    $sql = "SELECT id, setting_key, setting_value, description, created_at, updated_at
            FROM system_settings
            WHERE setting_key = ?";
    
    $setting = fetchSingleRow($sql, [$id]);
    
    if ($setting) {
        http_response_code(200);
        echo json_encode($setting);
    } else {
        http_response_code(404);
        echo json_encode(['error' => 'Banking setting not found']);
    }
}

function getBankingSettings() {
    // Get only banking-related settings
    $sql = "SELECT id, setting_key, setting_value, description, created_at, updated_at
            FROM system_settings
            WHERE setting_key LIKE 'bank_%' OR setting_key LIKE 'mpesa_%' OR setting_key = 'account_visibility'
            ORDER BY setting_key";
    
    $settings = executeQuery($sql);
    
    http_response_code(200);
    echo json_encode([
        'count' => count($settings),
        'settings' => $settings
    ]);
}

function handlePostRequest() {
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid JSON data']);
        return;
    }
    
    // Validate required fields
    $required_fields = ['setting_key', 'setting_value'];
    foreach ($required_fields as $field) {
        if (!isset($input[$field]) || empty($input[$field])) {
            http_response_code(400);
            echo json_encode(['error' => "Missing required field: $field"]);
            return;
        }
    }
    
    // Check if setting already exists
    $checkSql = "SELECT id FROM system_settings WHERE setting_key = ?";
    $existing = fetchSingleRow($checkSql, [$input['setting_key']]);
    
    if ($existing) {
        http_response_code(400);
        echo json_encode(['error' => 'Banking setting with this key already exists']);
        return;
    }
    
    // Insert setting
    $sql = "INSERT INTO system_settings (setting_key, setting_value, description) 
            VALUES (?, ?, ?)";
    
    $params = [
        $input['setting_key'],
        $input['setting_value'],
        $input['description'] ?? null
    ];
    
    try {
        beginTransaction();
        
        executeNonQuery($sql, $params);
        $setting_id = getDatabaseConnection()->lastInsertId();
        
        commitTransaction();
        
        http_response_code(201);
        echo json_encode([
            'message' => 'Banking setting created successfully',
            'setting_id' => $setting_id
        ]);
    } catch (Exception $e) {
        rollbackTransaction();
        throw $e;
    }
}

function handlePutRequest() {
    if (!isset($_GET['id'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Setting key is required']);
        return;
    }
    
    $key = $_GET['id'];
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid JSON data']);
        return;
    }
    
    // Prepare update data
    $update_fields = [];
    $update_params = [];
    
    // Add setting_value if provided and not null
    if (isset($input['setting_value']) && $input['setting_value'] !== null) {
        $update_fields[] = "setting_value = ?";
        $update_params[] = $input['setting_value'];
    }
    
    // Add description if provided and not null
    if (isset($input['description']) && $input['description'] !== null) {
        $update_fields[] = "description = ?";
        $update_params[] = $input['description'];
    }
    
    // Check if we have fields to update
    if (empty($update_fields)) {
        http_response_code(400);
        echo json_encode(['error' => 'No fields to update']);
        return;
    }
    
    // Build the SQL query
    $sql = "UPDATE system_settings SET " . implode(", ", $update_fields) . " WHERE setting_key = ?";
    $update_params[] = $key;
    
    try {
        beginTransaction();
        
        $rows_affected = executeNonQuery($sql, $update_params);
        
        if ($rows_affected === 0) {
            // Check if the setting actually exists
            $checkSql = "SELECT id FROM system_settings WHERE setting_key = ?";
            $existing = fetchSingleRow($checkSql, [$key]);
            if (!$existing) {
                http_response_code(404);
                echo json_encode(['error' => 'Banking setting not found']);
                rollbackTransaction();
                return;
            } else {
                http_response_code(200); // Still return success since no changes were needed
                echo json_encode(['message' => 'No changes were needed']);
                rollbackTransaction();
                return;
            }
        }
        
        commitTransaction();
        
        http_response_code(200);
        echo json_encode([
            'message' => 'Banking setting updated successfully'
        ]);
    } catch (Exception $e) {
        try {
            rollbackTransaction();
        } catch (Exception $rollbackException) {
            // Ignore rollback errors
        }
        throw $e;
    }
}

function handleDeleteRequest() {
    if (!isset($_GET['id'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Setting key is required']);
        return;
    }
    
    $key = $_GET['id'];
    
    // Check if the setting exists
    $checkSql = "SELECT id FROM system_settings WHERE setting_key = ?";
    $existing = fetchSingleRow($checkSql, [$key]);
    
    if (!$existing) {
        http_response_code(404);
        echo json_encode(['error' => 'Banking setting not found']);
        return;
    }
    
    // Delete the setting
    $sql = "DELETE FROM system_settings WHERE setting_key = ?";
    
    try {
        beginTransaction();
        
        executeNonQuery($sql, [$key]);
        
        commitTransaction();
        
        http_response_code(200);
        echo json_encode([
            'message' => 'Banking setting deleted successfully'
        ]);
    } catch (Exception $e) {
        try {
            rollbackTransaction();
        } catch (Exception $rollbackException) {
            // Ignore rollback errors
        }
        throw $e;
    }
}
?>