<?php
// MFT GROUP Audit Logs API
// This API provides audit log data for the frontend dashboards

// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Enable CORS for frontend access
header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Methods: GET, POST");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// Include database configuration with proper path resolution
$rootPath = dirname(__DIR__);
require_once $rootPath . '/database/config.php';

// Get request method
$method = $_SERVER['REQUEST_METHOD'] ?? 'GET';

// Ensure JSON header is always set
header("Content-Type: application/json; charset=UTF-8");

if ($method === 'POST') {
    handlePostRequest();
} else if ($method === 'GET') {
    handleGetRequest();
} else {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit;
}

function handlePostRequest() {
    // Handle creation of new audit logs with real-time notification
    $rawInput = file_get_contents('php://input');
    $input = json_decode($rawInput, true);
    
    if (!$input && !empty($rawInput)) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid JSON data']);
        error_log('Invalid JSON data received: ' . $rawInput);
        return;
    }
    
    try {
        // Extract audit log data
        $user_id = isset($input['user_id']) ? $input['user_id'] : null;
        $action = isset($input['action']) ? $input['action'] : '';
        $table_name = isset($input['table_name']) ? $input['table_name'] : '';
        $record_id = isset($input['record_id']) ? $input['record_id'] : null;
        $old_values = isset($input['old_values']) ? json_encode($input['old_values']) : null;
        $new_values = isset($input['new_values']) ? json_encode($input['new_values']) : null;
        $ip_address = isset($input['ip_address']) ? $input['ip_address'] : ($_SERVER['REMOTE_ADDR'] ?? '');
        $user_agent = isset($input['user_agent']) ? $input['user_agent'] : ($_SERVER['HTTP_USER_AGENT'] ?? '');
        
        // Insert audit log into database
        $sql = "INSERT INTO audit_logs (user_id, action, table_name, record_id, old_values, new_values, ip_address, user_agent) 
                VALUES (?, ?, ?, ?, ?, ?, ?, ?)";
        $params = [$user_id, $action, $table_name, $record_id, $old_values, $new_values, $ip_address, $user_agent];
        
        $log_id = executeInsert($sql, $params);
        
        // Get the inserted audit log with user details
        $log_sql = "SELECT a.id, a.user_id, a.action, a.table_name, a.record_id, a.old_values, a.new_values,
                           a.ip_address, a.user_agent, a.created_at, m.first_name, m.last_name, m.member_id as member_ref
                    FROM audit_logs a
                    LEFT JOIN members m ON a.user_id = m.id
                    WHERE a.id = ?";
        $log = fetchSingleRow($log_sql, [$log_id]);
        
        // Send real-time notification
        sendAuditTrailNotification($log);
        
        http_response_code(201);
        echo json_encode([
            'message' => 'Audit log created successfully',
            'id' => $log_id,
            'log' => $log
        ]);
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Internal server error: ' . $e->getMessage()]);
        error_log("Audit logs API error: " . $e->getMessage());
    }
}

function handleGetRequest() {
    // Ensure JSON header is set
    header("Content-Type: application/json; charset=UTF-8");
    
    if (isset($_GET['id'])) {
        // Get specific audit log
        getAuditLog($_GET['id']);
    } else {
        // Get all audit logs with optional filters
        getAuditLogs();
    }
}

function getAuditLog($id) {
    // Ensure JSON header is set
    header("Content-Type: application/json; charset=UTF-8");
    
    $sql = "SELECT a.id, a.user_id, a.action, a.table_name, a.record_id, a.old_values, a.new_values,
                   a.ip_address, a.user_agent, a.created_at, m.first_name, m.last_name, m.member_id as member_ref
            FROM audit_logs a
            LEFT JOIN members m ON a.user_id = m.id
            WHERE a.id = ?";
    
    $log = fetchSingleRow($sql, [$id]);
    
    if ($log) {
        http_response_code(200);
        echo json_encode($log);
    } else {
        http_response_code(404);
        echo json_encode(['error' => 'Audit log not found']);
    }
}

function getAuditLogs() {
    // Ensure JSON header is set
    header("Content-Type: application/json; charset=UTF-8");
    
    try {
        $action_filter = isset($_GET['action']) ? $_GET['action'] : null;
        $table_filter = isset($_GET['table']) ? $_GET['table'] : null;
        $user_filter = isset($_GET['user_id']) ? $_GET['user_id'] : null;
        $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 50;
        $offset = isset($_GET['offset']) ? (int)$_GET['offset'] : 0;
        
        // Ensure limit is reasonable
        if ($limit > 100) {
            $limit = 100;
        }
        
        // Ensure limit and offset are not negative
        $limit = max(1, $limit);
        $offset = max(0, $offset);
        
        $sql = "SELECT a.id, a.user_id, a.action, a.table_name, a.record_id, a.old_values, a.new_values,
                       a.ip_address, a.user_agent, a.created_at, m.first_name, m.last_name, m.member_id as member_ref
                FROM audit_logs a
                LEFT JOIN members m ON a.user_id = m.id";
        
        $params = [];
        $conditions = [];
        
        if ($action_filter) {
            $conditions[] = "a.action = ?";
            $params[] = $action_filter;
        }
        
        if ($table_filter) {
            $conditions[] = "a.table_name = ?";
            $params[] = $table_filter;
        }
        
        if ($user_filter) {
            $conditions[] = "a.user_id = ?";
            $params[] = $user_filter;
        }
        
        if (!empty($conditions)) {
            $sql .= " WHERE " . implode(" AND ", $conditions);
        }
        
        $sql .= " ORDER BY a.created_at DESC LIMIT ? OFFSET ?";
        $params[] = $limit;
        $params[] = $offset;
        
        $logs = executeQuery($sql, $params);
        
        // Get total count
        $count_sql = "SELECT COUNT(*) as total FROM audit_logs a";
        $count_params = [];
        
        // Add conditions to count query as well
        if (!empty($conditions)) {
            $count_sql .= " WHERE " . implode(" AND ", $conditions);
            // Add the condition parameters (but not the limit/offset)
            $count_params = array_slice($params, 0, count($params) - 2);
        }
        
        $count_result = fetchSingleRow($count_sql, $count_params);
        $total = $count_result ? $count_result['total'] : 0;
        
        http_response_code(200);
        echo json_encode([
            'count' => count($logs),
            'total' => $total,
            'offset' => $offset,
            'limit' => $limit,
            'logs' => $logs
        ]);
    } catch (Exception $e) {
        error_log("Error in getAuditLogs: " . $e->getMessage());
        http_response_code(500);
        echo json_encode(['error' => 'Internal server error: ' . $e->getMessage()]);
    }
}

// Function to send real-time notification for audit trail updates
function sendAuditTrailNotification($log) {
    try {
        // Include the WebSocket server functions if the file exists
        $websocketFile = __DIR__ . '/websocket-server.php';
        if (file_exists($websocketFile)) {
            require_once $websocketFile;
        }
        
        // Create a notification for the audit trail update
        $notification = [
            'type' => 'audit_trail_update',
            'data' => $log,
            'target' => 'admin'
        ];
        
        // Store notification in database
        $pdo = getDatabaseConnection();
        $type = isset($notification['type']) ? $notification['type'] : '';
        $data = isset($notification['data']) ? json_encode($notification['data']) : '{}';
        $target = isset($notification['target']) ? $notification['target'] : 'all';
        
        $sql = "INSERT INTO notifications (type, data, target) VALUES (?, ?, ?)";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([$type, $data, $target]);
        
        error_log("Audit trail notification sent: " . json_encode($notification));
    } catch (Exception $e) {
        error_log("Error sending audit trail notification: " . $e->getMessage());
    }
}
?>