<?php
// MFT GROUP Accounts API
// This API provides account data for the frontend dashboards

// Enable CORS for frontend access
header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");

// Include database configuration
require_once __DIR__ . '/../database/config.php';

// Get request method
$method = $_SERVER['REQUEST_METHOD'];

try {
    switch ($method) {
        case 'GET':
            handleGetRequest();
            break;
        case 'POST':
            handlePostRequest();
            break;
        case 'PUT':
            handlePutRequest();
            break;
        case 'DELETE':
            handleDeleteRequest();
            break;
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            break;
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Internal server error: ' . $e->getMessage()]);
}

function handleGetRequest() {
    if (isset($_GET['id'])) {
        // Get specific account
        getAccount($_GET['id']);
    } else if (isset($_GET['type'])) {
        // Get accounts by type
        getAccountsByType($_GET['type']);
    } else {
        // Get all accounts
        getAccounts();
    }
}

function getAccount($id) {
    $sql = "SELECT a.id, a.name, a.account_type_id, a.balance, a.description, a.is_active,
                   a.created_at, a.updated_at, t.name as account_type_name
            FROM accounts a
            JOIN account_types t ON a.account_type_id = t.id
            WHERE a.id = ?";
    
    $account = fetchSingleRow($sql, [$id]);
    
    if ($account) {
        http_response_code(200);
        echo json_encode($account);
    } else {
        http_response_code(404);
        echo json_encode(['error' => 'Account not found']);
    }
}

function getAccountsByType($type) {
    $sql = "SELECT a.id, a.name, a.account_type_id, a.balance, a.description, a.is_active,
                   a.created_at, a.updated_at, t.name as account_type_name
            FROM accounts a
            JOIN account_types t ON a.account_type_id = t.id
            WHERE t.name = ?
            ORDER BY a.name";
    
    $accounts = executeQuery($sql, [$type]);
    
    http_response_code(200);
    echo json_encode([
        'count' => count($accounts),
        'accounts' => $accounts
    ]);
}

function getAccounts() {
    $active_filter = isset($_GET['active']) ? (bool)$_GET['active'] : null;
    
    $sql = "SELECT a.id, a.name, a.account_type_id, a.balance, a.description, a.is_active,
                   a.created_at, a.updated_at, t.name as account_type_name
            FROM accounts a
            JOIN account_types t ON a.account_type_id = t.id";
    
    $params = [];
    
    if ($active_filter !== null) {
        $sql .= " WHERE a.is_active = ?";
        $params[] = $active_filter ? 1 : 0;
    }
    
    $sql .= " ORDER BY t.name, a.name";
    
    $accounts = executeQuery($sql, $params);
    
    http_response_code(200);
    echo json_encode([
        'count' => count($accounts),
        'accounts' => $accounts
    ]);
}

function handlePostRequest() {
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid JSON data']);
        return;
    }
    
    // Validate required fields
    $required_fields = ['name', 'account_type_id'];
    foreach ($required_fields as $field) {
        if (!isset($input[$field]) || empty($input[$field])) {
            http_response_code(400);
            echo json_encode(['error' => "Missing required field: $field"]);
            return;
        }
    }
    
    // Set default values
    if (!isset($input['balance'])) {
        $input['balance'] = 0.00;
    }
    
    if (!isset($input['is_active'])) {
        $input['is_active'] = true;
    }
    
    // Insert account
    $sql = "INSERT INTO accounts (name, account_type_id, balance, description, is_active) 
            VALUES (?, ?, ?, ?, ?)";
    
    $params = [
        $input['name'],
        $input['account_type_id'],
        $input['balance'],
        $input['description'] ?? null,
        $input['is_active'] ? 1 : 0
    ];
    
    try {
        beginTransaction();
        
        executeNonQuery($sql, $params);
        $account_id = getDatabaseConnection()->lastInsertId();
        
        commitTransaction();
        
        http_response_code(201);
        echo json_encode([
            'message' => 'Account created successfully',
            'account_id' => $account_id
        ]);
    } catch (Exception $e) {
        rollbackTransaction();
        throw $e;
    }
}

function handlePutRequest() {
    if (!isset($_GET['id'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Account ID is required']);
        return;
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid JSON data']);
        return;
    }
    
    // Update account
    $sql = "UPDATE accounts SET 
            name = ?, 
            account_type_id = ?, 
            balance = ?, 
            description = ?, 
            is_active = ?
            WHERE id = ?";
    
    $params = [
        $input['name'] ?? null,
        $input['account_type_id'] ?? null,
        $input['balance'] ?? null,
        $input['description'] ?? null,
        isset($input['is_active']) ? ($input['is_active'] ? 1 : 0) : null,
        $_GET['id']
    ];
    
    // Remove null values
    $fields = ['name', 'account_type_id', 'balance', 'description', 'is_active', 'id'];
    $filtered_params = [];
    $set_parts = [];
    
    for ($i = 0; $i < count($fields) - 1; $i++) {
        if ($params[$i] !== null) {
            $set_parts[] = $fields[$i] . " = ?";
            $filtered_params[] = $params[$i];
        }
    }
    
    if (empty($set_parts)) {
        http_response_code(400);
        echo json_encode(['error' => 'No fields to update']);
        return;
    }
    
    $sql = "UPDATE accounts SET " . implode(", ", $set_parts) . " WHERE id = ?";
    $filtered_params[] = $_GET['id'];
    
    try {
        beginTransaction();
        
        $rows_affected = executeNonQuery($sql, $filtered_params);
        
        if ($rows_affected === 0) {
            http_response_code(404);
            echo json_encode(['error' => 'Account not found']);
            return;
        }
        
        commitTransaction();
        
        http_response_code(200);
        echo json_encode([
            'message' => 'Account updated successfully'
        ]);
    } catch (Exception $e) {
        rollbackTransaction();
        throw $e;
    }
}

function handleDeleteRequest() {
    if (!isset($_GET['id'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Account ID is required']);
        return;
    }
    
    // Check if account has any transactions
    $check_sql = "SELECT COUNT(*) as count FROM payments WHERE account_id = ?";
    $result = fetchSingleRow($check_sql, [$_GET['id']]);
    
    if ($result['count'] > 0) {
        http_response_code(400);
        echo json_encode(['error' => 'Cannot delete account with existing transactions']);
        return;
    }
    
    $sql = "DELETE FROM accounts WHERE id = ?";
    
    try {
        beginTransaction();
        
        $rows_affected = executeNonQuery($sql, [$_GET['id']]);
        
        if ($rows_affected === 0) {
            http_response_code(404);
            echo json_encode(['error' => 'Account not found']);
            return;
        }
        
        commitTransaction();
        
        http_response_code(200);
        echo json_encode([
            'message' => 'Account deleted successfully'
        ]);
    } catch (Exception $e) {
        rollbackTransaction();
        throw $e;
    }
}
?>